/*
Filename    : ConnAPI.cs
Part of     : PC Suite connectivity API, C# examples
Description : Contains helper functions for using PC Suite Connectivity API methods
Version     : 1.1

This example is only to be used with PC Suite Connectivity API version 1.1.
Compability ("as is") with future versions is not quaranteed.

Copyright (c) 2005 Nokia Corporation.
 
This material, including but not limited to documentation and any related 
computer programs, is protected by intellectual property rights of Nokia 
Corporation and/or its licensors.
All rights are reserved. Reproducing, modifying, translating, or 
distributing any or all of this material requires the prior written consent 
of Nokia Corporation. Nokia Corporation retains the right to make changes 
to this material at any time without notice. A copyright license is hereby 
granted to download and print a copy of this material for personal use only.
No other license to any other intellectual property rights is granted. The 
material is provided "as is" without warranty of any kind, either express or 
implied, including without limitation, any warranty of non-infringement, 
merchantability and fitness for a particular purpose. In no event shall 
Nokia Corporation be liable for any direct, indirect, special, incidental, 
or consequential loss or damages, including but not limited to, lost profits 
or revenue,loss of use, cost of substitute program, or loss of data or 
equipment arising out of the use or inability to use the material, even if 
Nokia Corporation has been advised of the likelihood of such damages occurring.
*/

using System;
using System.Runtime.InteropServices;


namespace CONNAPI
{
    public enum Media : uint
    {
        All = 0x01,
        Irda = 0x02,
        Serial = 0x04,
        Bluetooth = 0x08,
        Usb = 0x10
    }

    /// <summary>
    /// Contains helper functions for using Connectivity API methods
    /// </summary>
    abstract class CONAPI_CONSTANTS
    {
        //=========================================================
        // Current Connectivity API version
        //
        public const int CONA_API_VERSION = 11;

        // Values used in callback registering methods
        public const uint CONAPI_REGISTER = 0x00;
        public const uint CONAPI_UNREGISTER = 0x01;

        // Media types
        //public const uint CONAPI_MEDIA_ALL = 0x01;
        //public const uint CONAPI_MEDIA_IRDA = 0x02;
        //public const uint CONAPI_MEDIA_SERIAL = 0x04;
        //public const uint CONAPI_MEDIA_BLUETOOTH = 0x08;
        //public const uint CONAPI_MEDIA_USB = 0x10;



        // Device callback status values
        public const uint CONAPI_DEVICE_LIST_UPDATED = 0x00;	// List is updated. No any specific information.
        public const uint CONAPI_DEVICE_ADDED = 0x01;	// A new device is added to the list.
        public const uint CONAPI_DEVICE_REMOVED = 0x02;	// Device is removed from the list.
        public const uint CONAPI_DEVICE_UPDATED = 0x04;	// Device is updated. A connection is added or removed
        // or device is renamed. Device still exist in the list.
        // Device callback info values
        public const uint CONAPI_CONNECTION_ADDED = 0x01;	// Note! HIBYTE == media, LOBYTE == CONAPI_CONNECTION_ADDED
        public const uint CONAPI_CONNECTION_REMOVED = 0x02;	// Note! HIBYTE == media, LOBYTE == CONAPI_CONNECTION_REMOVED
        public const uint CONAPI_DEVICE_RENAMED = 0x04;	// Friendly name of the device is changed

        //Options for CONADeleteFolder:
        public const uint CONA_DELETE_FOLDER_EMPTY = 0x00000000;
        public const uint CONA_DELETE_FOLDER_WITH_FILES = 0x00000001;

        //Direction options for CONACopyFile and CONAMoveFile:
        public const uint CONA_DIRECT_PHONE_TO_PC = 0x00000002;
        public const uint CONA_DIRECT_PC_TO_PHONE = 0x00000004;
        public const uint CONA_DIRECT_PHONE_TO_PHONE = 0x00000008;	// Not used at the moment.

        //Other options for CONACopyFile and CONAMoveFile:
        public const uint CONA_OVERWRITE = 0x00000010;
        public const uint CONA_RENAME = 0x00000020;	// Used only with CONACopyFile
        public const uint CONA_TRANSFER_ALL = 0x00000040;	// Not used at the moment.

        //Options for CONAFindBegin:
        public const uint CONA_FIND_USE_CACHE = 0x00000080;

        //Attribute defines for CONAPI_FOLDER_INFO and CONAPI_FILE_INFO structures:
        public const uint CONA_FPERM_READ = 0x00000100;	// Both structure
        public const uint CONA_FPERM_WRITE = 0x00000200;	// Both structure
        public const uint CONA_FPERM_DELETE = 0x00000400;	// Both structure
        public const uint CONA_FPERM_FOLDER = 0x00000800;	// Only for CONAPI_FOLDER_INFO
        public const uint CONA_FPERM_DRIVE = 0x00001000;	// Only for CONAPI_FOLDER_INFO

        public const uint CONA_DEFAULT_FOLDER = 0x00010000;  //Used only with CONAInstallApplication

        // Search definitions used with CONASearchDevices function:
        public const uint CONAPI_DEVICE_UNPAIRED = 0x00000001; // Device is not paired
        public const uint CONAPI_DEVICE_PAIRED = 0x00000002; // Device is paired
        public const uint CONAPI_DEVICE_PCSUITE_TRUSTED = 0x00000004; // Device is PC Suite trusted

        public const uint CONAPI_ALLOW_TO_USE_CACHE = 0x00001000; // Get all devices from cache if available
        public const uint CONAPI_GET_ALL_PHONES = 0x00002000; // Get all phones from target media
        public const uint CONAPI_GET_PAIRED_PHONES = 0x00004000; // Get all paired phones from target media
        public const uint CONAPI_GET_TRUSTED_PHONES = 0x00008000; // Get all PC Suite trusted phones from target media.

        // Definitions used with CONAChangeDeviceTrustedState function:
        public const uint CONAPI_PAIR_DEVICE = 0x00000100; // Pair device
        public const uint CONAPI_UNPAIR_DEVICE = 0x00000200; // Unpair device
        public const uint CONAPI_SET_PCSUITE_TRUSTED = 0x00000400; // Set device to PC Suite trusted 
        public const uint CONAPI_SET_PCSUITE_UNTRUSTED = 0x00000800; // Remove PC Suite trusted information.

        // FSFunction values:
        public const uint CONARefreshDeviceMemoryValuesNtf = 0x00000001;
        public const uint CONASetCurrentFolderNtf = 0x00000002;
        public const uint CONAFindBeginNtf = 0x00000004;
        public const uint CONACreateFolderNtf = 0x00000008;
        public const uint CONADeleteFolderNtf = 0x00000010;
        public const uint CONARenameFolderNtf = 0x00000020;
        public const uint CONAGetFileInfoNtf = 0x00000040;
        public const uint CONADeleteFileNtf = 0x00000080;
        public const uint CONAMoveFileNtf = 0x00000100;
        public const uint CONACopyFileNtf = 0x00000200;
        public const uint CONARenameFileNtf = 0x00000400;
        public const uint CONAReadFileNtf = 0x00000800;
        public const uint CONAWriteFileNtf = 0x00001000;
        public const uint CONAConnectionLostNtf = 0x00002000;
        public const uint CONAInstallApplicationNtf = 0x00004000;

        //The next define values used to define which type of struct is used:
        public const uint CONA_APPLICATION_TYPE_SIS = 0x00000001; // Use when struct type is CONAPI_APPLICATION_SIS
        public const uint CONA_APPLICATION_TYPE_JAVA = 0x00000002; // Use when struct type is CONAPI_APPLICATION_JAVA
    };

    abstract class CONAPI_ERRORS
    {
        public const uint CONA_OK = 0x00000000;	// Everything ok
        // Common errorCode codes:
        public const uint ECONA_INIT_FAILED = 0x80100000;	// DLL initialization failed
        public const uint ECONA_INIT_FAILED_COM_INTERFACE = 0x80100002;	// Failed to get connection to ServiceLayer.
        public const uint ECONA_NOT_INITIALIZED = 0x80100004;	// API is not initialized
        public const uint ECONA_UNSUPPORTED_API_VERSION = 0x80100005;	// Failed, not supported API version
        public const uint ECONA_NOT_SUPPORTED_MANUFACTURER = 0x80100006;	// Failed, not supported manufacturer

        public const uint ECONA_UNKNOWN_ERROR = 0x80100010;	// Failed, unknown errorCode
        public const uint ECONA_UNKNOWN_ERROR_DEVICE = 0x80100011;	// Failed, unknown errorCode from Device
        public const uint ECONA_INVALID_POINTER = 0x80100012;	// Required pointer is invalid
        public const uint ECONA_INVALID_PARAMETER = 0x80100013;	// Invalid Parameter value 
        public const uint ECONA_INVALID_HANDLE = 0x80100014;	// Invalid HANDLE
        public const uint ECONA_NOT_ENOUGH_MEMORY = 0x80100015;	// Memory allocation failed in PC
        public const uint ECONA_WRONG_THREAD = 0x80100016;	// Failed, Called interface was marshalled for a different thread.

        public const uint ECONA_CANCELLED = 0x80100020;	// Operation cancelled by ConnectivityAPI-User
        public const uint ECONA_NOTHING_TO_CANCEL = 0x80100021;	// No running functions
        public const uint ECONA_FAILED_TIMEOUT = 0x80100022;	// Operation failed because of timeout
        public const uint ECONA_NOT_SUPPORTED_DEVICE = 0x80100023;	// Device do not support operation
        public const uint ECONA_NOT_SUPPORTED_PC = 0x80100024;	// ConnectivityAPI do not support operation (not implemented)

        // Device manager and device connection releated errorCode:
        public const uint ECONA_DEVICE_NOT_FOUND = 0x80200000;	// Given phone is not connected (refresh device list)
        public const uint ECONA_NO_CONNECTION_VIA_MEDIA = 0x80200001;	// Phone is connected but not via given Media
        public const uint ECONA_NO_CONNECTION_VIA_DEVID = 0x80200002;	// Phone is not connected with given DevID
        public const uint ECONA_INVALID_CONNECTION_TYPE = 0x80200003;	// Connection type was invalid
        public const uint ECONA_NOT_SUPPORTED_CONNECTION_TYPE = 0x80200004;	// Device do not support connection type
        public const uint ECONA_CONNECTION_BUSY = 0x80200005;	// Other application is reserved connection
        public const uint ECONA_CONNECTION_LOST = 0x80200006;	// Connection is lost to Device
        public const uint ECONA_CONNECTION_REMOVED = 0x80200007;	// Connection removed, other application is reserved connection.
        public const uint ECONA_CONNECTION_FAILED = 0x80200008;	// Connection failed, unknown reason
        public const uint ECONA_SUSPEND = 0x80200009;	// Connection removed, PC goes suspend state
        public const uint ECONA_NAME_ALREADY_EXISTS = 0x8020000A;	// Friendly name already exist

        // File System errors:
        public const uint ECONA_ALL_LISTED = 0x80300000;	// All items are listed
        public const uint ECONA_MEMORY_FULL = 0x80300001;	// Device memory full

        // File System errorCode for file functions:
        public const uint ECONA_FILE_NAME_INVALID = 0x80400001;	// File name includes invalid characters in Device or PC
        public const uint ECONA_FILE_NAME_TOO_LONG = 0x80400002;	// Max unicode charaters in File name (includes current/target path) 
        // is limited to 256 charaters in device. (256 charaters includes 
        // two backlashs in begin of the path and one end of the path.)

        public const uint ECONA_FILE_ALREADY_EXIST = 0x80400003;	// File is already exits in Device or PC
        public const uint ECONA_FILE_NOT_FOUND = 0x80400004;	// File does not exits in Device or PC
        public const uint ECONA_FILE_NO_PERMISSION = 0x80400005;	// Not allow to perform required operation to file in Device 
        public const uint ECONA_FILE_COPYRIGHT_PROTECTED = 0x80400006;	// Not allow to perform required operation to file in Device
        public const uint ECONA_FILE_BUSY = 0x80400007;	// Other application is reserved file in Device or PC
        public const uint ECONA_FILE_TOO_BIG_DEVICE = 0x80400008;	// Device reject the operation because file size is too big
        public const uint ECONA_FILE_TYPE_NOT_SUPPORTED = 0x80400009;	// Device reject the operation because file unsupported type
        public const uint ECONA_FILE_NO_PERMISSION_ON_PC = 0x8040000A;	// Not allow to perform required operation to file in PC 
        public const uint ECONA_FILE_EXIST = 0x8040000B;	// File move or rename: If is copied but ConnAPI can not remove source file.

        // File System errorCode for folder functions:
        public const uint ECONA_INVALID_DATA_DEVICE = 0x80500000;	// Device's folder contain invalid data
        public const uint ECONA_CURRENT_FOLDER_NOT_FOUND = 0x80500001;	// Current/Target folder is invalid in device (e.g MMC card removed). 
        public const uint ECONA_FOLDER_PATH_TOO_LONG = 0x80500002;	// Current/Target folder max unicode charaters count is limited to 
        // 245 charaters in device. (245 charaters includes two backlashs in 
        // begin of the path and one end of the path)

        public const uint ECONA_FOLDER_NAME_INVALID = 0x80500003;	// Folder name includes invalid characters in Device or PC
        public const uint ECONA_FOLDER_ALREADY_EXIST = 0x80500004;	// Folder is already exits in target folder
        public const uint ECONA_FOLDER_NOT_FOUND = 0x80500005;	// Folder not found (PC/Device)
        public const uint ECONA_FOLDER_NO_PERMISSION = 0x80500006;	// Not allow to perform required operation to folder in Devic
        public const uint ECONA_FOLDER_NOT_EMPTY = 0x80500007;	// Not allow to perform required operation because folder is not empty
        public const uint ECONA_FOLDER_NO_PERMISSION_ON_PC = 0x80500008;	// Not allow to perform required operation to folder in PC

        // Application Installation:
        public const uint ECONA_DEVICE_INSTALLER_BUSY = 0x80600000;	// Can not start Device's installer

        // Device paring releated errors:
        public const uint ECONA_DEVICE_PAIRING_FAILED = 0x80200100;	// Failed, pairing failed
        public const uint ECONA_DEVICE_PASSWORD_WRONG = 0x80200101;	// Failed, wrong password on device. 
        public const uint ECONA_DEVICE_PASSWORD_INVALID = 0x80200102;	// Failed, password includes invalid characters or missing.

        // Converts given errorCode code to a string
        static public String CONAError2String(uint error)
        {
            switch (error)
            {
                // Common errorCode codes:
                case CONA_OK: return "CONA_OK: Succedeed";
                case ECONA_INIT_FAILED: return "ECONA_INIT_FAILED: DLL initialization failed";
                case ECONA_INIT_FAILED_COM_INTERFACE: return "ECONA_INIT_FAILED_COM_INTERFACE: DLL initialization failed";
                case ECONA_NOT_INITIALIZED: return "ECONA_NOT_INITIALIZED: API is not initialized";
                case ECONA_UNSUPPORTED_API_VERSION: return "ECONA_UNSUPPORTED_API_VERSION: API version not supported";
                case ECONA_NOT_SUPPORTED_MANUFACTURER: return "ECONA_NOT_SUPPORTED_MANUFACTURER: Manufacturer is not supported";

                case ECONA_UNKNOWN_ERROR: return "ECONA_UNKNOWN_ERROR: Failed, unknown errorCode";
                case ECONA_UNKNOWN_ERROR_DEVICE: return "ECONA_UNKNOWN_ERROR_DEVICE: Failed, unknown errorCode from Device";
                case ECONA_INVALID_POINTER: return "ECONA_INVALID_POINTER: Required pointer is invalid";
                case ECONA_INVALID_PARAMETER: return "ECONA_INVALID_PARAMETER: Invalid parameter value";
                case ECONA_INVALID_HANDLE: return "ECONA_INVALID_HANDLE: Invalid HANDLE";
                case ECONA_NOT_ENOUGH_MEMORY: return "ECONA_NOT_ENOUGH_MEMORY: Memory allocation failed in PC";
                case ECONA_WRONG_THREAD: return "ECONA_WRONG_THREAD: Called interface was marshalled for a different thread";

                case ECONA_CANCELLED: return "ECONA_CANCELLED: Operation cancelled by ConnectivityAPI-User";
                case ECONA_NOTHING_TO_CANCEL: return "ECONA_NOTHING_TO_CANCEL: No running functions";
                case ECONA_FAILED_TIMEOUT: return "ECONA_FAILED_TIMEOUT: Operation failed because of timeout";
                case ECONA_NOT_SUPPORTED_DEVICE: return "ECONA_NOT_SUPPORTED_DEVICE: Device does not support operation";
                case ECONA_NOT_SUPPORTED_PC: return "ECONA_NOT_SUPPORTED_PC: Connectivity API does not support operation (not implemented)";

                // Device manager and device connection releated errorCode:
                case ECONA_DEVICE_NOT_FOUND: return "ECONA_DEVICE_NOT_FOUND: Given phone is not connected (refresh device list)";
                case ECONA_NO_CONNECTION_VIA_MEDIA: return "ECONA_NO_CONNECTION_VIA_MEDIA: Phone is connected but not via given Media";
                case ECONA_NO_CONNECTION_VIA_DEVID: return "ECONA_NO_CONNECTION_VIA_DEVID: Phone is not connected with given DevID";
                case ECONA_INVALID_CONNECTION_TYPE: return "ECONA_INVALID_CONNECTION_TYPE: Connection type was invalid";
                case ECONA_NOT_SUPPORTED_CONNECTION_TYPE: return "ECONA_NOT_SUPPORTED_CONNECTION_TYPE: Device does not support connection type";
                case ECONA_CONNECTION_BUSY: return "ECONA_CONNECTION_BUSY: Other application is reserved connection";
                case ECONA_CONNECTION_LOST: return "ECONA_CONNECTION_LOST: Connection is lost to Device";
                case ECONA_CONNECTION_REMOVED: return "ECONA_CONNECTION_REMOVED: Connection removed, other application has reserved connection.";
                case ECONA_CONNECTION_FAILED: return "ECONA_CONNECTION_FAILED: Connection failed, unknown reason";
                case ECONA_SUSPEND: return "ECONA_SUSPEND: Connection removed, PC goes suspend state";
                case ECONA_NAME_ALREADY_EXISTS: return "ECONA_NAME_ALREADY_EXISTS: Friendly name already exist";

                // File System errors:
                case ECONA_ALL_LISTED: return "ECONA_ALL_LISTED: All items are listed";
                case ECONA_MEMORY_FULL: return "ECONA_MEMORY_FULL: Device memory full";

                // File System errorCode for file functions:
                case ECONA_FILE_NAME_INVALID: return "ECONA_FILE_NAME_INVALID: File name includes invalid characters in Device or PC";
                case ECONA_FILE_NAME_TOO_LONG: return "ECONA_FILE_NAME_TOO_LONG: File name includes too many characters in Device or PC";
                case ECONA_FILE_ALREADY_EXIST: return "ECONA_FILE_ALREADY_EXIST: File already exists in Device or PC";
                case ECONA_FILE_NOT_FOUND: return "ECONA_FILE_NOT_FOUND: File does not exist in Device or PC";
                case ECONA_FILE_NO_PERMISSION: return "ECONA_FILE_NO_PERMISSION: Not allowed to perform required operation to file in Device or PC";
                case ECONA_FILE_COPYRIGHT_PROTECTED: return "ECONA_FILE_COPYRIGHT_PROTECTED: Not allowed to perform required operation to file in Device or PC";
                case ECONA_FILE_BUSY: return "ECONA_FILE_BUSY: Other application has reserved file in Device or PC";
                case ECONA_FILE_TOO_BIG_DEVICE: return "ECONA_FILE_TOO_BIG_DEVICE: Device rejected the operation because file size is too big";
                case ECONA_FILE_TYPE_NOT_SUPPORTED: return "ECONA_FILE_TYPE_NOT_SUPPORTED: Device rejected the operation because file is unsupported type";
                case ECONA_FILE_NO_PERMISSION_ON_PC: return "ECONA_FILE_NO_PERMISSION_ON_PC: ECONA_FILE_NO_PERMISSION_ON_PC";

                // File System errorCode for folder functions:
                case ECONA_INVALID_DATA_DEVICE: return "ECONA_INVALID_DATA_DEVICE: Device's folder contains invalid data";
                case ECONA_CURRENT_FOLDER_NOT_FOUND: return "ECONA_CURRENT_FOLDER_NOT_FOUND: Current folder is invalid in device (e.g MMC card removed).";
                case ECONA_FOLDER_PATH_TOO_LONG: return "ECONA_FOLDER_PATH_TOO_LONG: Current folder max unicode charaters count is limited to 255.";
                case ECONA_FOLDER_NAME_INVALID: return "ECONA_FOLDER_NAME_INVALID: Folder name includes invalid characters in Device or PC";
                case ECONA_FOLDER_ALREADY_EXIST: return "ECONA_FOLDER_ALREADY_EXIST: Folder already exists in target folder";
                case ECONA_FOLDER_NOT_FOUND: return "ECONA_FOLDER_NOT_FOUND: Folder does not exist in target folder";
                case ECONA_FOLDER_NO_PERMISSION: return "ECONA_FOLDER_NO_PERMISSION: Not allowed to perform required operation to folder in Device";
                case ECONA_FOLDER_NOT_EMPTY: return "ECONA_FOLDER_NOT_EMPTY: Not allowed to perform required operation because folder is not empty";
                case ECONA_FOLDER_NO_PERMISSION_ON_PC: return "ECONA_FOLDER_NO_PERMISSION_ON_PC: Not allowed to perform required operation to folder in PC";
                // Application Installation:
                case ECONA_DEVICE_INSTALLER_BUSY: return "ECONA_DEVICE_INSTALLER_BUSY: Can not start Device's installer";

                // Bluetooth pairing
                case ECONA_DEVICE_PAIRING_FAILED: return "ECONA_DEVICE_PAIRING_FAILED: Pairing failed";
                case ECONA_DEVICE_PASSWORD_WRONG: return "ECONA_DEVICE_PASSWORD_WRONG: Wrong password on device";
                case ECONA_DEVICE_PASSWORD_INVALID: return "ECONA_DEVICE_PASSWORD_INVALID: Password includes invalid characters or missing";

                default: return "Undefined error code"; // shouldn't occur
            }
        }

        static public void ShowError(String strDescription, uint dwReason)
        {
            String str = String.Format("{0}\nErrorCodeor: {1:x}\n{2}", strDescription, dwReason, CONAError2String(dwReason));
            System.Windows.Forms.MessageBox.Show(str);

        }
    };

    public class ConnAPI
    {
        public ConnAPI()
        {
        }

        //===================================================================
        // GetMediaString: 
        // Converts the given media constant to a string format
        //===================================================================
        public static String GetMediaString(uint media)
        {
            if (media == (uint)Media.Irda)
            {
                return "IrDA";
            }
            else if (media == (uint)Media.Serial)
            {
                return "SerialNumber";
            }
            else if (media == (uint)Media.Bluetooth)
            {
                return "Bluetooth";
            }
            else if (media == (uint)Media.Usb)
            {
                return "USB";
            }
            else
            {
                return "Unknown Media";
            }
        }

        //===================================================================
        // GetTrustedStateString: 
        // Converts the given trusted state to string format
        //===================================================================
        public static String GetTrustedStateString(uint uiTrustedState)
        {
            string strRet = "";

            if (CONAPI_IS_DEVICE_PAIRED(uiTrustedState))
            {
                strRet += "Paired, ";
            }
            else if (CONAPI_IS_DEVICE_UNPAIRED(uiTrustedState))
            {
                strRet += "Unpaired, ";
            }

            if (CONAPI_IS_DEVICE_PCSUITE_TRUSTED(uiTrustedState))
            {
                strRet += "Trusted";
            }
            else
            {
                strRet += "Untrusted";
            }

            return strRet;
        }


        // Search functions used to check device's trusted/paired state:
        // These functions implements the same functionality as PC Suite Connectivity API
        // C++ macros CONAPI_IS_DEVICE_PAIRED, CONAPI_IS_DEVICE_UNPAIRED and
        // CONAPI_IS_DEVICE_PCSUITE_TRUSTED
        //===================================================================
        // CONAPI_IS_DEVICE_PAIRED:
        // Returns true, if given device status is paired
        //===================================================================
        public static bool CONAPI_IS_DEVICE_PAIRED(uint iState)
        {
            return ((iState & CONNAPI.CONAPI_CONSTANTS.CONAPI_DEVICE_PAIRED) != 0);
        }

        //===================================================================
        // CONAPI_IS_DEVICE_UNPAIRED:
        // Returns true, if given device status is unpaired
        //===================================================================
        public static bool CONAPI_IS_DEVICE_UNPAIRED(uint iState)
        {
            return ((iState & CONNAPI.CONAPI_CONSTANTS.CONAPI_DEVICE_UNPAIRED) != 0);
        }

        //===================================================================
        // CONAPI_IS_DEVICE_PCSUITE_TRUSTED:
        // Returns true, if given device status is "PC Suite trusted"
        //===================================================================
        public static bool CONAPI_IS_DEVICE_PCSUITE_TRUSTED(uint iState)
        {
            return ((iState & CONNAPI.CONAPI_CONSTANTS.CONAPI_DEVICE_PCSUITE_TRUSTED) != 0);
        }


        // Connection info structure
        [StructLayout(LayoutKind.Sequential)]
        public unsafe struct CONAPI_CONNECTION_INFO
        {
            public uint dwDeviceID;
            public uint dwMedia;
            [MarshalAs(UnmanagedType.LPWStr)]
            public String pstrDeviceName;
            [MarshalAs(UnmanagedType.LPWStr)]
            public String pstrAddress;
            public uint dwState;
        };

        // Device info structure
        [StructLayout(LayoutKind.Sequential)]
        public unsafe struct CONAPI_DEVICE
        {
            [MarshalAs(UnmanagedType.LPWStr)]
            public String pstrSerialNumber;
            [MarshalAs(UnmanagedType.LPWStr)]
            public String pstrFriendlyName;
            [MarshalAs(UnmanagedType.LPWStr)]
            public String pstrModel;
            [MarshalAs(UnmanagedType.LPWStr)]
            public String pstrManufacturer;
            public uint dwNumberOfItems;
            public IntPtr pItems;
        };

        // Folder info structure
        [StructLayout(LayoutKind.Sequential)]
        public unsafe struct CONAPI_FOLDER_INFO
        {
            public char* pstrName;
            public uint dwAttributes;
            public FILETIME tFolderTime;
            public char* pstrLabel;
            public char* pstrMemoryType;
        };

        // File info structure
        [StructLayout(LayoutKind.Sequential)]
        public unsafe struct CONAPI_FILE_INFO
        {
            public char* pstrName;
            public uint dwAttributes;
            public FILETIME tFiletime;
            public uint dwFileSize;
            public char* pstrMIMEType;
        };

        //The struct for java applications
        [StructLayout(LayoutKind.Sequential)]
        public unsafe struct CONAPI_APPLICATION_JAVA
        {
            public char* pstrFileNameJad;
            public char* pstrFileNameJar;
        };

        //The struct for sis applications
        [StructLayout(LayoutKind.Sequential)]
        public unsafe struct CONAPI_APPLICATION_SIS
        {
            public char* pstrFileNameSis;
        };

        // Device notify callback function prototype:
        public delegate uint CallBack(int dwStatus, [MarshalAs(UnmanagedType.LPWStr)]String pstrSerialNumber);
        // File System callback function prototype
        public delegate uint FSCallBack(int dwOperation, int dwStatus, int dwTransferredBytes, int dwAllBytes);
        // SearchDevice callback function prototype
        public delegate uint SearchCallBack(int dwState, IntPtr pConnInfoStructure);

        //=========================================================
        // CONAInitialize
        //
        // Description:
        //	CONAInitialize initializes the Connectivity API -library 
        //	and connects to the ServiceLayer COM interfaces. This 
        //	function must be called before any other ConnectivityAPI 
        //	functions, and it must be called once per every ConnectivityAPI 
        //	instance. A multithreaded application needs to call this 
        //	function only once. 
        //	
        //	All successful calls of this method needs corresponding number of 
        //  CONAUninitialize calls.
        //
        //  This ConnectivityAPI supported version is 1.1 and the 
        //  value for that is 11. An API user must give a number 	
        //  value of the ConnectivyAPI version in the parameter. 
        //  If the API user gives a bigger version number than what 
        //  the installed ConnectivityAPI supports, the function fails 
        //  with the errorCode code ECONA_UNSUPPORTED_API_VERSION.
        //  
        // Parameters:
        //	dwVersion		[in]     Version number, value should be CONA_API_VERSION 
        //	pstrManufacturer[in]     Target manufacturer string. If used, 
        //			only the target manufacturer devices are available. 
        //			E.g. the String for Nokia devices is "Nokia". 
        //			If the string is NULL, all devices are available.
        //	pValue			[in,out] Reserved for future use, the value must be NULL. 
        //
        // Return values:
        // CONA_OK
        // ECONA_INIT_FAILED
        // ECONA_UNSUPPORTED_API_VERSION
        // ECONA_NOT_SUPPORTED_MANUFACTURER
        // ECONA_INVALID_PARAMETER 
        // ECONA_INVALID_POINTER
        [DllImport("ConnAPI.dll")]
        public static extern uint CONAInitialize(int dwVersion, [MarshalAs(UnmanagedType.LPWStr)]String pstrManufacturer, IntPtr pValue);
        //=========================================================

        //=========================================================
        // CONAUninitialize
        //
        // Description:
        //	CONAUninitialize deconnects the ConnectivityAPI from 
        //  the ServiceLayer and frees the used resources. This 
        //  function must be called just before application termination.
        //
        //	All calls of this method needs corresponding number of 
        //  successful CONAInitialize calls. Ohterwise all resourses
        //  and memories won't be released.
        //
        // Parameters:
        //	dwValue			[in] Reserved for future use, the value must be zero.  
        //
        // Return values:
        // CONA_OK
        // ECONA_NOT_INITIALIZED
        // ECONA_UNKNOWN_ERROR
        // ECONA_INVALID_PARAMETER 
        //
        [DllImport("ConnAPI.dll")]
        public static extern uint CONAUninitialize(int dwValue);
        //=========================================================

        //=========================================================
        // CONAOpenDM
        //
        // Description:
        //	CONAOpenDM opens the device management connection. 
        //	It returns a handle to the device management context.
        //	This must be used in all device management functions.
        //
        // Parameters:
        //	phDMHandle		[out] Device manager handle
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_POINTER
        // ECONA_NOT_INITIALIZED
        // ECONA_NOT_ENOUGH_MEMORY
        // ECONA_INIT_FAILED_COM_INTERFACE
        // ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public unsafe static extern uint CONAOpenDM(IntPtr* phDMHandle);
        //=========================================================

        //=========================================================
        // CONACloseDM
        //
        // Description:
        //	CONACloseDM is used to close device management connection. 
        //	This must be called for each device management handle 
        //	when it is not needed anymore.
        //
        // Parameters:
        //	hDMHandle		[in] Device manager handle
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_POINTER
        // ECONA_INVALID_HANDLE
        // ECONA_NOT_INITIALIZED
        // ECONA_UNKNOWN_ERROR
        // 
        [DllImport("ConnAPI.dll")]
        public static extern uint CONACloseDM(IntPtr phDMHandle);
        //=========================================================

        //=========================================================
        // CONAGetDeviceCount
        //
        // Description:
        //	CONAGetDeviceCount puts the count of devices into the 
        //	given DWORD. 
        //
        // Parameters:
        //	hDMHandle		[in]  Device manager handle
        //	pdwCount		[out] Pointer to the variable receiving 
        //						  the count of devices.
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_POINTER
        // ECONA_INVALID_HANDLE
        // ECONA_NOT_INITIALIZED
        // ECONA_UNKNOWN_ERROR
        // 
        [DllImport("ConnAPI.dll")]
        public unsafe static extern uint CONAGetDeviceCount(IntPtr phDMHandle, int* dwDeviceCount);
        //=========================================================

        //=========================================================
        // CONAGetDevices
        //
        // Description:
        //	CONAGetDevices allocates and sets devices information to 
        //	allocate the CONAPI_DEVICE structs and returns the count 
        //	of the used CONAPI_DEVICE structs.  The allocated memory 
        //	MUST be freed by calling the CONAFreeDeviceStructure function. 
        //	
        //	Every CONAPI_DEVICE struct includes CONAPI_CONNECT_INFO 
        //	sturct(s), which include the media type information. 
        //	If the device is connected via different medias at the 
        //	same time, there is an individual CONAPI_CONNECT_INFO for 
        //	each media type.

        //
        // Parameters:
        //  hDMHandle		[in]  Device manager handle
        //  pdwCount		[in,out] In: Number of allocated CONAPI_DEVICE structs.
        //							 Out: Number of used CONAPI_DEVICE structs.
        //  pDevices		[out] Pointer to receiving CONAPI_DEVICE structures.
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_POINTER
        // ECONA_INVALID_PARAMETER
        // ECONA_INVALID_HANDLE
        // ECONA_NOT_INITIALIZED
        // ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public unsafe static extern uint CONAGetDevices(IntPtr hDMHandle, int* dwDeviceCount, IntPtr pDevices);
        //=========================================================

        //=========================================================
        // CONAFreeDeviceStructure
        //
        // Description:
        //	CONAFreeDeviceStructure releases the memory, which 
        //	ConnectivitAPI is allocated inside the CONAPI_DEVICE structs.
        //
        // Parameters:
        //	dwCount			[in] Number of the used CONAPI_DEVICE structs. 
        //		The same number value if CONAGetDevices function is used. 
        //		The value must be the number of the used structs (pdwCount). 
        //		If the CONAGetDevice function is used, the value must be 1.
        //	pDevices		[in] Pointer to the device's struct list.
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_POINTER
        //	ECONA_INVALID_PARAMETER
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static extern uint CONAFreeDeviceStructure(int dwCount, IntPtr pDevices);

        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAFreeFileInfoStructure(CONAPI_FILE_INFO* pFileInfo);

        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAFreeFolderInfoStructure(CONAPI_FOLDER_INFO* pFolderInfo);
        //=========================================================

        //=========================================================
        // CONARefreshDeviceList
        //
        // Description:
        //	CONARefreshDeviceList function starts to refresh the 
        //	Device list. All changes will be notified via notifications.
        //
        // Parameters:
        //	hDMHandle		[in] Device manager handle
        //	dwValue			[in] Reserved for future use. Must be zero.
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_PARAMETER
        // ECONA_INVALID_HANDLE
        // ECONA_NOT_INITIALIZED
        // ECONA_UNKNOWN_ERROR
        [DllImport("ConnAPI.dll")]
        public static extern uint CONARefreshDeviceList(IntPtr hDMHandle, int uiValue);

        //=========================================================
        // CONARenameFriendlyName
        //
        // Description:
        //	Renames the device's friendly name. Current device's 
        //	friendly name string is in the CONAPI_DEVICE struct; 
        //	When the function returns a successful result, the 
        //	Connectivity API sends a DeviceList nofication with the 
        //	CONAPI_DEVICE_UPDATED value to all Connectivity API-users. 
        //
        // Parameters:
        //	hDMHandle			[in] Device manager handle
        //	pstrSerialNumber	[in] SerialNumber number of the device.
        //	pstrNewFriendlyName [in] New Device Friendly Name .
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_POINTER
        // ECONA_INVALID_HANDLE
        // ECONA_NAME_ALREADY_EXISTS
        // ECONA_NOT_INITIALIZED
        // ECONA_NOT_SUPPORTED_PC
        // ECONA_UNKNOWN_ERROR
        [DllImport("ConnAPI.dll")]
        public static extern uint CONARenameFriendlyName(IntPtr hDMHandle,
            [MarshalAs(UnmanagedType.LPWStr)]String pstrSerialNumber,
            [MarshalAs(UnmanagedType.LPWStr)]String pstrNewFriendlyName);

        //=========================================================
        // CONARegisterNotifyCallback
        //
        // Description:
        //	CONARegisterNotifyCallback is used to register or unregister 
        //	a callback function for the Connectivity API device list 
        //	notifications.
        //
        // Parameters:
        //	hDMHandle		[in] Device manager handle
        //	dwState			[in] Used to define the action:
        //						 CONAPI_REGISTER used in registeration
        //						 CONAPI_REGISTER used in removing the registeration
        //	pfnNotify		[in] Function pointer of the call back method
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_POINTER
        // ECONA_INVALID_HANDLE
        // ECONA_INVALID_PARAMETER
        // ECONA_NOT_INITIALIZED
        // ECONA_UNKNOWN_ERROR

        [DllImport("ConnAPI.dll")]
        public static extern uint CONARegisterNotifyCallback(IntPtr hDMHandle, uint dwState, CallBack pfnNotify);
        //=========================================================

        //=========================================================
        // CONAInstallApplication 
        //
        // Description:
        //	CONAInstallApplication installs the given application on 
        //	the device. FS's CONACancel function can be used to cancel 
        //	this function. The application file name(s) must be given 
        //	in the CONAPI_APPLICATION_SIS or CONAPI_APPLICATION_JAVA 
        //	structure. 
        //	
        //	The function also checks the amount of free memory from 
        //	the phone before the installation. If there is not enough 
        //	memory left in the target memory drive, the function fails 
        //	with the errorCode code ECONA_MEMORY_FULL.
        //	
        //	If the Application file type is sis, the CONAPI_APPLICATION_SIS 
        //	structure must be used. The dwAppicationType parameter's 
        //	value must be CONA_APPLICATION_TYPE_SIS.
        //	
        //	If the Application file type is jar, the CONAPI_APPLICATION_JAVA 
        //	structure must be used. The dwAppicationType parameter's 
        //	value must be CONA_APPLICATION_TYPE_JAVA.
        //
        // Parameters:
        //	hFSHandle			[in] File System handle
        //	dwApplicationType	[in] Used struct type: CONA_APPLICATION_TYPE_SIS or 
        //											   CONA_APPLICATION_TYPE_JAVA
        //	pApplicationStruct	[in] Pointer to CONAPI_APPLICATION_SIS or 
        //							 CONAPI_APPLICATION_JAVA struct.
        //	dwOptions			[in] Options: 
        //		CONA_DEFAULT_FOLDER: Copies the application to the device's default 
        //		application folder automatically and starts the device's installer 
        //		(if required to do so).Target path or current folder is not used.  
        //		CONA_OVERWRITE:	Overwrites the application file if it already 
        //		exists in the target folder.
        //
        //	pstrSourcePath		[in] Source folder path on the PC. The value must be NULL, 
        //							 if path is given with file name(s) in Application structure.
        //	pstrTargetPath   	[in] Target folder.If NULL, current folder is used.
        //							 If CONA_DEFAULT_FOLDER is used,Application will 
        //							 be installed to device's default application folder. 
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_HANDLE
        // ECONA_INVALID_PARAMETER
        // ECONA_CONNECTION_BUSY
        // ECONA_CONNECTION_LOST
        // ECONA_INVALID_DATA_DEVICE
        // ECONA_MEMORY_FULL
        // ECONA_CURRENT_FOLDER_NOT_FOUND
        // ECONA_FOLDER_PATH_TOO_LONG
        // ECONA_FOLDER_NOT_FOUND
        // ECONA_FOLDER_NO_PERMISSION_ON_PC
        // ECONA_FILE_TOO_BIG_DEVICE
        // ECONA_FILE_NAME_INVALID
        // ECONA_FILE_NAME_TOO_LONG
        // ECONA_FILE_TYPE_NOT_SUPPORTED
        // ECONA_FILE_NOT_FOUND
        // ECONA_FILE_ALREADY_EXIST
        // ECONA_FILE_NO_PERMISSION
        // ECONA_FILE_NO_PERMISSION_ON_PC
        // ECONA_FILE_BUSY
        // ECONA_DEVICE_INSTALLER_BUSY
        // ECONA_CANCELLED
        // ECONA_FAILED_TIMEOUT
        // ECONA_FILE_TYPE_NOT_SUPPORTED
        // ECONA_NOT_SUPPORTED_DEVICE
        // ECONA_UNKNOWN_ERROR_DEVICE
        // ECONA_NOT_INITIALIZED
        // ECONA_NOT_SUPPORTED_MANUFACTURER
        // ECONA_UNKNOWN_ERROR
        // 
        [DllImport("ConnAPI.dll")]
        public unsafe static extern uint CONAInstallApplication(IntPtr hFSHandle, uint dwApplicationType, IntPtr pApplicationStruct, uint dwOptions, char* pstrSourcePath, char* pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONAOpenFS
        //
        // Description:
        //	CONAOpenFS opens a file system (FS) connection to the target 
        //	device. The function sets the FS Connection handle, which 
        //	identifies different client connections. It will be placed 
        //	into the given pointer. 
        //	
        //	If the media type is CONAPI_MEDIA_ALL, the ServiceLayer selects 
        //	the used media type automatically by applying the following rule:
        //	If there is an open FS connection to the device, it will be used. 
        //	Otherwise the ServiceLayer tries to open the FS connection in the 
        //	following order: USB, Seria, Irda and BT (if the device is 
        //	connected via medias). The function also sets the DeviceID, 
        //	which can be used to identify the used media type.
        //	
        //	If the SerialNumber Number is an empty string, the function tries to 
        //	open the FS connection by using the pdwDeviceID parameter. The 
        //	pdwMedia parameter is not used in the exemplary case.
        //
        // Parameters:
        //	pstrSerialNumber [in]     The device's serialNumber number, which must be set.
        //	pdwMedia		 [in,out] In: Media type, which defines the target
        //								 media for the Device's FS connection.
        //							  Out:If the value is CONAPI_MEDIA_ALL, the 
        //								 target media is selected automatically 
        //								 and the used media type is returned.
        //	phFSHandle		 [out]	  Handle to the opened FS connection
        //	pdwDeviceID		 [in,out] In: Device ID. Used only when the SerialNumber 
        //								 number is empty string.
        //							  Out: Returned Device ID value of the connected device.
        //
        // Return values:
        // CONA_OK
        // ECONA_DEVICE_NOT_FOUND
        // ECONA_NO_CONNECTION_VIA_MEDIA
        // ECONA_NOT_SUPPORTED_DEVICE
        // ECONA_CONNECTION_FAILED
        // ECONA_CONNECTION_BUSY
        // ECONA_CONNECTION_LOST
        // ECONA_FAILED_TIMEOUT
        // ECONA_INVALID_POINTER
        // ECONA_NOT_INITIALIZED
        // ECONA_NOT_SUPPORTED_MANUFACTURER
        // ECONA_NOT_ENOUGH_MEMORY
        // ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public unsafe static extern uint CONAOpenFS([MarshalAs(UnmanagedType.LPWStr)]String pstrSerialNumber, uint* pdwMedia, IntPtr* phFSHandle, uint* pdwDeviceID);
        //=========================================================

        //=========================================================
        // CONACloseFS
        //
        // Description:
        //	CONACloseFS closes the given file system connection.
        //
        // Parameters:
        //	hFSHandle	[in] Existing file system handle
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static extern uint CONACloseFS(IntPtr hFSHandle);
        //=========================================================

        //=========================================================
        // CONARegisterFSNotifyCallback
        //
        // Description:
        // CONARegisterFSNotifyCallback registers or unregisters the 
        // callback function for the file functions' status notifications.
        // 
        // During file operations, the connectionAPI calls this function 
        // with the parameter's state (0-100%), connection handle, and 
        // operation code.

        //
        // Parameters:
        //	hFSHandle	[in] File System handle
        //	dwState		[in] Register flag:
        //					  CONAPI_REGISTER:   Registers a callback function
        //					  CONAPI_UNREGISTER: Unregisters a callback function
        //	pFSNotify	[in] Pointer to a function.
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_POINTER
        //	ECONA_INVALID_PARAMETER
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static extern uint CONARegisterFSNotifyCallback(IntPtr hFSHandle, uint dwState, FSCallBack pFSNotify);
        //=========================================================

        //=========================================================
        // CONARefreshDeviceMemoryValues
        //
        // Description:
        //	CONARefreshDeviceMemoryValues refreshes the memory values 
        // from the device. The function caches the memory values, so 
        // it must be called if it is required to get the latest information.
        //
        // Parameters:
        //	hFSHandle	[in] File system handle
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static extern uint CONARefreshDeviceMemoryValues(IntPtr hFSHandle);
        //=========================================================

        //=========================================================
        // CONAGetMemoryValues
        //
        // Description:
        //	CONAGetDeviceMemory returns info about the given device memory type.
        //
        // Parameters:
        //	hFSHandle		[in]  File System handle.
        //	pstrMemoryType	[in]  Inquired device memory.E.g. "DEV", "MMC", 
        //                        or "APPL".If the value is NULL, the used value is "DEV".
        //	pdlFree			[out] Pointer to the variable receiving the 
        //						  information on the amount of free memory.
        //						  If not available, the value is -1.
        //	pdlTotal		[out] Pointer to the variable receiving 
        //						  the information on the total memory.
        //						  If not available, the value is -1.
        //	pdlUsed			[out] Pointer to the variable receiving 
        //						  the information on the used memory.
        //						  If not available, the value is -1.
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_POINTER
        //	ECONA_INVALID_PARAMETER
        //	ECONA_NOT_SUPPORTED_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAGetMemoryValues(IntPtr hFSHandle, char* pstrMemoryType, Int64* pdlFree, Int64* pdlTotal, Int64* pdlUsed);
        //=========================================================

        //=========================================================
        // CONAFindBegin
        //
        // Description:
        // CONAFindBegin starts find files and subfolders from currect 
        // folder or target path. If target path is used, path must be 
        // absolute path.Find option:
        // CONA_FIND_USE_CACHE	  Use cache if it is available.
        //
        // Parameters:
        //	hFSHandle		[in]  File System handle
        //	dwFindOptions	[in]  Find options:If CONA_FIND_USE_CACHE option
        //						  is not used, value must be 0.
        //	pFileHandle		[out] Pointer to find handle
        //	pstrTargetPath	[in]  Target path. If the path is NULL, 
        //						  the current folder is used 
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_INVALID_POINTER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //  ECONA_NOT_ENOUGH_MEMORY
        //	ECONA_UNKNOWN_ERROR
        //	ECONA_FOLDER_PATH_TOO_LONG
        //	ECONA_FOLDER_NAME_INVALID_DEVICE
        //	ECONA_FOLDER_NO_PERMISSION
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAFindBegin(IntPtr hFSHandle, uint dwFindOptions, IntPtr* phFindHandle, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONAFindNextFolder
        //
        // Description:
        // CONAFindNextFolder returns the folder information for one 
        // folder and continues the search to the next folder. The 
        // folder information is returned with the CONAPI_FOLDER_INFO 
        // structure. The allocated memory MUST be freed by calling 
        // the CONAFreeFolderInfoStructure function.
        //
        // Parameters:
        //	hFindHandle		[in]  Find handle
        //	pFolderInfo		[out] Pointer to the folder's info struct.
        //
        // Return values:
        //	CONA_OK
        //	ECONA_ALL_LISTED
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_POINTER
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAFindNextFolder(IntPtr hFindHandle, CONAPI_FOLDER_INFO* pFolderInfo);
        //=========================================================

        //=========================================================
        // CONAFindNextFile
        //
        // Description:
        // CONAFindNextFile returns file information for one file, 
        // and continues the search to the next file. File information 
        // is returned with the CONAPI_FILE_INFO structure. Allocated 
        // memory MUST be freed by calling CONAFreeFileInfoStructure function.
        //
        // Parameters:
        //	hFindHandle		[in]  Find handle
        //	pFileInfo		[out] Pointer to the file's info struct.
        //
        // Return values:
        //	CONA_OK
        //	ECONA_ALL_LISTED
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_POINTER
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAFindNextFile(IntPtr hFindHandle, CONAPI_FILE_INFO* pFileInfo);
        //=========================================================

        //=========================================================
        // CONACreateFolder
        //
        // Description:
        // CONACreateFolder creates a new folder onto the device. 
        // The function fails if the folder already exists on the device.
        //
        // Parameters:
        //	hFSHandle		[in] File System handle
        //	pstrFolderName	[in] New folder name string
        //	pstrTargetPath	[in] Target path. If path is NULL, current 
        //						 folder is used. 
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //	ECONA_FOLDER_PATH_TOO_LONG
        //	ECONA_FOLDER_NAME_INVALID_DEVICE
        //	ECONA_FOLDER_ALREADY_EXIST
        //	ECONA_FOLDER_NO_PERMISSION
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONACreateFolder(IntPtr hFSHandle, [MarshalAs(UnmanagedType.LPWStr)]String pstrFolderName, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONARenameFolder
        //
        // Description:
        // CONARenameFolder renames a folder on the device. Only an 
        // empty folder can be renamed at the moment.

        //
        // Parameters:
        //	hFSHandle		  [in] File System handle
        //	pstrFolderName	  [in] Folder name string.
        //	pstrFolderNameNew [in] Folder's new name string.
        //	pstrTargetPath	  [in] Target path. If the path is NULL, 
        //                         current folder is used. 
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //  ECONA_FOLDER_PATH_TOO_LONG
        //	ECONA_FOLDER_ALREADY_EXIST
        //	ECONA_FOLDER_NOT_FOUND
        //	ECONA_FOLDER_NOT_EMPTY
        //	ECONA_FOLDER_NO_PERMISSION
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONARenameFolder(IntPtr hFSHandle, [MarshalAs(UnmanagedType.LPWStr)]String pstrFolderName, [MarshalAs(UnmanagedType.LPWStr)]String pstrFolderNameNew, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONARenameFile
        //
        // Description:
        //	CONARenameFile renames the given file.
        //
        // Parameters:
        //	hFSHandle		[in] File System handle.
        //	pstrFileName	[in] Target file name.
        //	pstrNewFileName	[in] New file name.
        //	pstrTargetPath	[in] Target path. If the path is NULL, 
        //						 the current folder is used. 
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_MEMORY_FULL
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //	ECONA_FOLDER_PATH_TOO_LONG
        //	ECONA_FILE_NAME_INVALID
        //	ECONA_FILE_NAME_TOO_LONG
        //	ECONA_FILE_NOT_FOUND
        //	ECONA_FILE_ALREADY_EXIST
        //	ECONA_FILE_COPYRIGHT_PROTECTED
        //	ECONA_FILE_NO_PERMISSION
        //	ECONA_FILE_BUSY
        //	ECONA_FILE_TYPE_NOT_SUPPORTED
        //  ECONA_FILE_EXIST
        //	ECONA_NOT_SUPPORTED_DEVICE
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_NOT_ENOUGH_MEMORY
        //	ECONA_UNKNOWN_ERROR
        //	
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONARenameFile(IntPtr hFSHandle, [MarshalAs(UnmanagedType.LPWStr)]String pstrFileName, [MarshalAs(UnmanagedType.LPWStr)]String pstrNewFileName, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONADeleteFolder
        //
        // Description:
        // CONADeleteFolder deletes a folder from the device. This 
        // function does not delete the folder if the folder includes 
        // any subfolder(s)! Only an empty folder and a folder that 
        // contains files can be deleted. If the folder does not include 
        // a DELETE-permission, the function fails.
        //
        // Note: No rollback option! Deleted data cannot be restored!
        //
        // Parameters:
        //	hFSHandle		[in] File System handle
        //	pstrFolderName	[in] Folder name string
        //	dwDeleteOptions	[in] Delete options. Possible values: 
        //			CONA_DELETE_FOLDER_EMPTY:		Only an empty folder can be deleted.
        //			CONA_DELETE_FOLDER_WITH_FILES:  Only a folder that contains file(s) can be deleted.
        //	pstrTargetPath	[in] Target path. If path is NULL, current
        //						 folder is used.  
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //	ECONA_FOLDER_PATH_TOO_LONG
        //	ECONA_FOLDER_NOT_FOUND
        //	ECONA_FOLDER_NOT_EMPTY
        //	ECONA_FOLDER_NO_PERMISSION
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONADeleteFolder(IntPtr hFSHandle, [MarshalAs(UnmanagedType.LPWStr)]String pstrFolderName, uint dwDeleteOptions, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONADeleteFile
        //
        // Description:
        //	CONADeleteFile deletes a file from the device's folder. 
        //	Note: No rollback option! Deleted data cannot be restored!
        //
        // Parameters:
        //	hFSHandle		[in] File System handle
        //	pstrFileName	[in] File name 
        //	pstrTargetPath	[in] Target path. If the path is NULL, 
        //                       the current folder is used.  
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //	ECONA_FILE_NOT_FOUND
        //	ECONA_FILE_NO_PERMISSION
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONADeleteFile(IntPtr hFSHandle, [MarshalAs(UnmanagedType.LPWStr)]String pstrFileName, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONAMoveFile
        //
        // Description:
        //	CONAMoveFile copy file to target folder on PC or Device 
        //  from Device or PC source folder and removes a file from 
        //  the source folder on the Device or PC. 
        //
        //  Note: No rollback option! If an errorCode occurs during the
        //	execution of the command (e.g. connection lost: file from 
        //  target folder will be deleted if it is changed), the 
        //  changes that have been already made are not cancelled.
        //  
        //  If the current folder is used, the path to the device's 
        //  source or target folder does not need to be set. PC side 
        //  path can also include the target/source file name. If path 
        //  to device's target or source folder is given, it cannot 
        //  include file name.
        //  
        //
        // Parameters:
        //	hFSHandle		[in] File System handle
        //	dwOptions		[in] Direction options:
        //		CONA_DIRECT_PHONE_TO_PC: Moves a file from the device to the PC.
        //		CONA_DIRECT_PC_TO_PHONE: Moves a file from the PC to the 
        //						 Other options:
        //		CONA_OVERWRITE:  Moves the file even if the target folder contains 
        //						 a file of the same name. The file will be replaced.
        //	pstrFileName	[in] File name. Can be NULL, if file name 
        //						 is given with PC path. 
        //	pstrSourcePath	[in] Source folder path. Value can include
        //						 also the source file name, if option
        //						 CONA_DIRECT_PC_TO_PHONE is used (PC path)
        //	pstrTargetPath	[in] Target folder path. Value can include  
        //						 also the target file name, if option 
        //						 CONA_DIRECT_PHONE_TO_PC is used (PC path)
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_MEMORY_FULL
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //	ECONA_FOLDER_PATH_TOO_LONG
        //	ECONA_FOLDER_NOT_FOUND
        //	ECONA_FILE_TOO_BIG_DEVICE
        //	ECONA_FILE_NAME_INVALID
        //	ECONA_FILE_NAME_TOO_LONG
        //	ECONA_FILE_NOT_FOUND
        //	ECONA_FILE_ALREADY_EXIST
        //	ECONA_FILE_COPYRIGHT_PROTECTED
        //	ECONA_FILE_NO_PERMISSION
        //	ECONA_FOLDER_NO_PERMISSION
        //	ECONA_FILE_NO_PERMISSION_ON_PC
        //  ECONA_FOLDER_NO_PERMISSION_ON_PC
        //	ECONA_FILE_BUSY
        //	ECONA_FILE_TYPE_NOT_SUPPORTED
        //	ECONA_FILE_EXIST
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_NOT_SUPPORTED_DEVICE
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_NOT_ENOUGH_MEMORY
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAMoveFile(IntPtr hFSHandle, uint dwOptions, [MarshalAs(UnmanagedType.LPWStr)]String pstrFileName, [MarshalAs(UnmanagedType.LPWStr)]String pstrSourcePath, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONACopyFile
        //
        // Description:
        //	CONACopyFile copy file to target folder on PC or Device 
        //	from Device or PC source folder 
        //	
        //	If current folder is used, path to device folder do not 
        //	need to set. Current folder can be used for source and 
        //	target location. PC side path can also include the 
        //	target/source file name. If path to device's target or 
        //	source folder is given, it cannot include file name.
        //	
        //  Note: No rollback option! If an errorCode occurs during the
        //	execution of the command (e.g. connection lost: file from 
        //  target folder will be deleted if it is changed), the 
        //  changes that have been already made are not cancelled.
        //	
        //
        // Parameters:
        //	hFSHandle		[in] File System handle
        //	dwOptions		[in] Copy options, Direction options:
        //			CONA_DIRECT_PC_TO_PHONE: Copies a file from the PC onto the device.
        //			CONA_DIRECT_PHONE_TO_PC: Copies a file from the device onto the PC.
        //						Other options:
        //			CONA_RENAME: Renames the file if it already exists in the target folder.
        //			CONA_OVERWRITE: Overwrites the file if it already exists in the target folder.
        //	pstrFileName	[in] File name. Can be NULL, if file name 
        //						 is given with PC path. 
        //	pstrSourcePath	[in] Source folder path. Value can include
        //						 also the source file name, if option
        //						 CONA_DIRECT_PC_TO_PHONE is used (PC path)
        //	pstrTargetPath	[in] Target folder path. Value can include  
        //						 also the target file name, if option 
        //						 CONA_DIRECT_PHONE_TO_PC is used (PC path)
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_HANDLE
        //	ECONA_INVALID_PARAMETER
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_INVALID_DATA_DEVICE
        //	ECONA_MEMORY_FULL
        //	ECONA_CURRENT_FOLDER_NOT_FOUND
        //	ECONA_FOLDER_PATH_TOO_LONG
        //	ECONA_FOLDER_NOT_FOUND
        //	ECONA_FILE_TOO_BIG_DEVICE
        //	ECONA_FILE_NAME_INVALID
        //	ECONA_FILE_NAME_TOO_LONG
        //	ECONA_FILE_NOT_FOUND
        //	ECONA_FILE_ALREADY_EXIST
        //	ECONA_FILE_COPYRIGHT_PROTECTED
        //	ECONA_FILE_NO_PERMISSION
        //	ECONA_FOLDER_NO_PERMISSION
        //	ECONA_FILE_NO_PERMISSION_ON_PC
        //	ECONA_FOLDER_NO_PERMISSION_ON_PC
        //	ECONA_FILE_BUSY
        //	ECONA_FILE_TYPE_NOT_SUPPORTED
        //	ECONA_FILE_NO_PERMISSION_ON_PC
        //	ECONA_CANCELLED
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_NOT_SUPPORTED_DEVICE
        //	ECONA_UNKNOWN_ERROR_DEVICE
        //	ECONA_NOT_INITIALIZED
        //	ECONA_NOT_ENOUGH_MEMORY
        //	ECONA_UNKNOWN_ERROR
        //
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONACopyFile(IntPtr hFSHandle, uint dwOptions, [MarshalAs(UnmanagedType.LPWStr)]String pstrFileName, [MarshalAs(UnmanagedType.LPWStr)]String pstrSourcePath, [MarshalAs(UnmanagedType.LPWStr)]String pstrTargetPath);
        //=========================================================

        //=========================================================
        // CONASearchDevices 
        //
        // Description:
        //	CCONASearchDevices functions search all devices from the target media. 
        //  The target media that can be used is Bluetooth at the moment.
        //
        //  CONASearchDevices allocates and sets devices information to 
        //  CONAPI_CONNECTION_INFO structs and returns pointer to structs.  
        //  Connectivity API-user MUST releases the returned pointer by calling the 
        //  CONAFreeConnectionInfoStructures function. 
        //
        //	Every CONAPI_CONNECTION_INFO struct includes the media and device information.
        //	The struct's dwState parameter defines the paired and trusted information from 
        //  device. It has the following values:
        //  Parameter value					Description					Macros for check the values 
        //																(If value is true, macro returns 1)
        //	CONAPI_DEVICE_UNPAIRED			Device in not paired.		CONAPI_IS_DEVICE_UNPAIRED(dwState)
        //	CONAPI_DEVICE_PAIRED			Device is paired.			CONAPI_IS_DEVICE_PAIRED(dwState)
        //	CONAPI_DEVICE_PCSUITE_TRUSTED	Device is PC Suite trusted.	CONAPI_IS_PCSUITE_TRUSTED(dwState)
        //
        //	Connectivity API can add more values afterwards so Connectivity API-user should 
        //	always use defined macros to check those values!

        //
        // Parameters:
        //	hDMHandle				[in] Existing device management handle.
        //	dwSearchOptions			[in] Search options values:
        //			CONAPI_MEDIA_BLUETOOTH: Get devices from bluetooth media. 
        //				This value must be used.
        //			CONAPI_ALLOW_TO_USE_CACHE: Get all devices from cache if available. 
        //				If cache is not available function fails with errorCode: ECONA_CACHE_IS_NOT_AVAILABLE.
        //				This value is optional and can be used with other values.
        //			One of the next values can be used at the time:
        //			CONAPI_GET_ALL_PHONES: Get all phones from target media. Includes unpaired, 
        //				paired and PC Suite trusted phones. 
        //			CONAPI_GET_PAIRED_PHONES:Get all paired phones from target media. Includes 
        //				paired (and PC Suite trusted) phones. 
        //			CONAPI_GET_TRUSTED_PHONES:Get all PC Suite trusted phones from target media. 
        //				Includes all PC Suite trusted phones, which are paired.
        //	dwSearchTime			[in]  Maximum search time in seconds. Note: Bluetooth device
        //			discovery can takes several minutes if there are a lot of devices on range!
        //	pfnSearchNotify			[in]  Pointer to search notification callback function. Value
        //			can be NULL if notification is not needed.
        //	pdwNumberOfStructures	[out] Number of CONAPI_CONNECTION_INFO structures. 
        //	ppConnInfoStructures	[out] Pointer to CONAPI_CONNECTION_INFO structure(s). 

        //	hDMHandle			[in] Device manager handle
        //	pstrSerialNumber	[in] SerialNumber number of the device.
        //	pstrNewFriendlyName [in] New Device Friendly Name .
        //
        // Return values:
        // CONA_OK
        // ECONA_INVALID_POINTER
        // ECONA_INVALID_HANDLE
        // ECONA_DEVICE_NOT_FOUND
        // ECONA_FAILED_TIMEOUT
        // ECONA_NO_CONNECTION_VIA_MEDIA
        // ECONA_MEDIA_IS_NOT_WORKING
        // ECONA_CACHE_IS_NOT_AVAILABLE
        // ECONA_SUSPEND
        // ECONA_NOT_ENOUGH_MEMORY
        // ECONA_NOT_INITIALIZED
        // ECONA_NOT_SUPPORTED_PC
        // ECONA_CANCELLED
        // ECONA_UNKNOWN_ERROR
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONASearchDevices(
            IntPtr hDMHandle,
            uint uiSearchOptions,
            uint uiSearchTime,
            SearchCallBack pfnSearchNotify,
            uint* puiNumberOfStructures,
            IntPtr* ppConnInfoStructures);
        /*
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONASearchDevices(
            IntPtr hDMHandle,
            uint uiSearchOptions,
            uint uiSearchTime,
            SearchCallBack pfnSearchNotify,
            uint *puiNumberOfStructures,
            CONAPI_CONNECTION_INFO** ppConnInfoStructures );
            */
        //=========================================================

        //=========================================================
        // CONAFreeConnectionInfoStructures
        //
        // Description:
        //	CONAFreeDeviceStructure releases the CONAPI_CONNECTION_INFO structs, 
        //	which CONASearchDevices function is allocated.
        //
        // Parameters:
        //	dwNumberOfStructures	[in] Number of CONAPI_CONNECTION_INFO structures.
        //	ppConnInfoStructures	[in] Pointer to CONAPI_CONNECTION_INFO structure(s).
        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_POINTER
        //	ECONA_INVALID_PARAMETER
        //	ECONA_UNKNOWN_ERROR
        [DllImport("ConnAPI.dll")]
        public static unsafe extern uint CONAFreeConnectionInfoStructures(
            uint uiNumberOfStructures,
            IntPtr* ppConnInfoStructures);
        //=========================================================

        //=========================================================
        // CONAChangeDeviceTrustedState
        //
        // Description:
        //	CONAChangeDeviceTrustedState functions changes device's 
        //	trusted state. It has the following operation values:
        //	Value							Description
        //	CONAPI_PAIR_DEVICE				Pair device.
        //	CONAPI_UNPAIR_DEVICE			Unpair device from Bluetooth stack.
        //	CONAPI_SET_PCSUITE_TRUSTED		Set device to PC Suite trusted state.
        //		System recognise the device and sets it to trusted state. Connectivity 
        //		API starts sends device notification from device.Note: Device must be 
        //		paired. Value can be used with CONAPI_PAIR_DEVICE. 
        //	CONAPI_SET_PCSUITE_UNTRUSTED	Remove PC Suite trusted information from 
        //		System. Connectivity API does not send device notification from device 
        //		anymore.Note: Device can be paired or unpaired state. Value can be used 
        //		with CONAPI_UNPAIR_DEVICE.

        //
        // Parameters:
        //	hDMHandle			[in] Existing device management handle.
        //	dwTrustedOperation	[in] Operation values: 
        //								CONAPI_PAIR_DEVICE
        //								CONAPI_UNPAIR_DEVICE
        //								CONAPI_SET_PCSUITE_TRUSTED 
        //								CONAPI_SET_PCSUITE_UNTRUSTED
        //	pstrAddress			[in] Device address. If device is connected via Bluetooth 
        //				media, address must be Device BT address (see pstrAddress parameter 
        //				from CONAPI_CONNECTION_INFO Structure)
        //	pstrPassword		[in] Password string for pairing. String can include only 
        //				the numbers (0-9) characters.Value used only with CONAPI_PAIR_DEVICE 
        //				operation. With other operations value should be NULL.
        //	pstrName			[in] Reserved for future use, the value must be NULL.

        //
        // Return values:
        //	CONA_OK
        //	ECONA_INVALID_POINTER
        //	ECONA_INVALID_HANDLE
        //	ECONA_DEVICE_NOT_FOUND
        //	ECONA_NOT_SUPPORTED_DEVICE
        //	ECONA_CONNECTION_FAILED
        //	ECONA_CONNECTION_FAILED
        //	ECONA_CONNECTION_BUSY
        //	ECONA_CONNECTION_LOST
        //	ECONA_DEVICE_PAIRING_FAILED
        //	ECONA_DEVICE_ PASSWORD_WRONG
        //	ECONA_DEVICE_ PASSWORD_INVALID
        //	ECONA_FAILED_TIMEOUT
        //	ECONA_NO_CONNECTION_VIA_MEDIA
        //	ECONA_MEDIA_IS_NOT_WORKING
        //	ECONA_SUSPEND
        //	ECONA_NOT_ENOUGH_MEMORY
        //	ECONA_NOT_INITIALIZED
        //	ECONA_NOT_SUPPORTED_PC
        //	ECONA_UNKNOWN_ERROR
        [DllImport("ConnAPI.dll")]
        public static extern uint CONAChangeDeviceTrustedState(
            IntPtr hDMHandle,
            uint uiTrustedOperation,
            [MarshalAs(UnmanagedType.LPWStr)]String pstrAddress,
            [MarshalAs(UnmanagedType.LPWStr)]String pstrPassword,
            [MarshalAs(UnmanagedType.LPWStr)]String pstrName
        );
        //=========================================================
    }
}
