using System;

namespace MosTech.Nokia
{
    internal class ErrorHelper
    {
        public static string GetMessage(ErrorCode errorCode)
        {
            switch (errorCode)
            {
                // Common errorCode codes:
                case ErrorCode.OK: return "OK: Succedeed";
                case ErrorCode.InitFailed: return "InitFailed: DLL initialization failed";
                case ErrorCode.InitFailedComInterface: return "InitFailedComInterface: DLL initialization failed";
                case ErrorCode.NotInitialized: return "NotInitialized: API is not initialized";
                case ErrorCode.UnsupportedApiVersion: return "UnsupportedApiVersion: API version not supported";
                case ErrorCode.NotSupportedManufacturer: return "NotSupportedManufacturer: Manufacturer is not supported";

                case ErrorCode.UnknownError: return "UnknownError: Failed, unknown errorCode";
                case ErrorCode.UnknownErrorDevice: return "UnknownErrorDevice: Failed, unknown errorCode from Device";
                case ErrorCode.InvalidPointer: return "InvalidPointer: Required pointer is invalid";
                case ErrorCode.InvalidParameter: return "InvalidParameter: Invalid parameter value";
                case ErrorCode.InvalidHandle: return "InvalidHandle: Invalid HANDLE";
                case ErrorCode.NotEnoughMemory: return "NotEnoughMemory: Memory allocation failed in PC";
                case ErrorCode.WrongThread: return "WrongThread: Called interface was marshalled for a different thread";

                case ErrorCode.Cancelled: return "Cancelled: Operation cancelled by ConnectivityAPI-User";
                case ErrorCode.NothingToCancel: return "NothingToCancel: No running functions";
                case ErrorCode.FailedTimeout: return "FailedTimeout: Operation failed because of timeout";
                case ErrorCode.NotSupportedDevice: return "NotSupportedDevice: Device does not support operation";
                case ErrorCode.NotSupportedPc: return "NotSupportedPc: Connectivity API does not support operation (not implemented)";

                // Device manager and device connection releated errorCode:
                case ErrorCode.DeviceNotFound: return "DeviceNotFound: Given phone is not connected (refresh device list)";
                case ErrorCode.NoConnectionViaMedia: return "NoConnectionViaMedia: Phone is connected but not via given Media";
                case ErrorCode.NoConnectionViaDevid: return "NoConnectionViaDevid: Phone is not connected with given DevID";
                case ErrorCode.InvalidConnectionType: return "InvalidConnectionType: Connection type was invalid";
                case ErrorCode.NotSupportedConnectionType: return "NotSupportedConnectionType: Device does not support connection type";
                case ErrorCode.ConnectionBusy: return "ConnectionBusy: Other application is reserved connection";
                case ErrorCode.ConnectionLost: return "ConnectionLost: Connection is lost to Device";
                case ErrorCode.ConnectionRemoved: return "ConnectionRemoved: Connection removed, other application has reserved connection.";
                case ErrorCode.ConnectionFailed: return "ConnectionFailed: Connection failed, unknown reason";
                case ErrorCode.Suspend: return "Suspend: Connection removed, PC goes suspend state";
                case ErrorCode.NameAlreadyExists: return "NameAlreadyExists: Friendly name already exist";

                // File System errors:
                case ErrorCode.AllListed: return "AllListed: All items are listed";
                case ErrorCode.MemoryFull: return "MemoryFull: Device memory full";

                // File System errorCode for file functions:
                case ErrorCode.FileNameInvalid: return "FileNameInvalid: File name includes invalid characters in Device or PC";
                case ErrorCode.FileNameTooLong: return "FileNameTooLong: File name includes too many characters in Device or PC";
                case ErrorCode.FileAlreadyExist: return "FileAlreadyExist: File already exists in Device or PC";
                case ErrorCode.FileNotFound: return "FileNotFound: File does not exist in Device or PC";
                case ErrorCode.FileNoPermission: return "FileNoPermission: Not allowed to perform required operation to file in Device or PC";
                case ErrorCode.FileCopyrightProtected: return "FileCopyrightProtected: Not allowed to perform required operation to file in Device or PC";
                case ErrorCode.FileBusy: return "FileBusy: Other application has reserved file in Device or PC";
                case ErrorCode.FileTooBigDevice: return "FileTooBigDevice: Device rejected the operation because file size is too big";
                case ErrorCode.FileTypeNotSupported: return "FileTypeNotSupported: Device rejected the operation because file is unsupported type";
                case ErrorCode.FileNoPermissionOnPc: return "FileNoPermissionOnPc: FILENOPERMISSIONONPC";

                // File System errorCode for folder functions:
                case ErrorCode.InvalidDataDevice: return "InvalidDataDevice: Device's folder contains invalid data";
                case ErrorCode.CurrentFolderNotFound: return "CurrentFolderNotFound: Current folder is invalid in device (e.g MMC card removed).";
                case ErrorCode.FolderPathTooLong: return "FolderPathTooLong: Current folder max unicode charaters count is limited to 255.";
                case ErrorCode.FolderNameInvalid: return "FolderNameInvalid: Folder name includes invalid characters in Device or PC";
                case ErrorCode.FolderAlreadyExist: return "FolderAlreadyExist: Folder already exists in target folder";
                case ErrorCode.FolderNotFound: return "FolderNotFound: Folder does not exist in target folder";
                case ErrorCode.FolderNoPermission: return "FolderNoPermission: Not allowed to perform required operation to folder in Device";
                case ErrorCode.FolderNotEmpty: return "FolderNotEmpty: Not allowed to perform required operation because folder is not empty";
                case ErrorCode.FolderNoPermissionOnPc: return "FolderNoPermissionOnPc: Not allowed to perform required operation to folder in PC";
                // Application Installation:
                case ErrorCode.DeviceInstallerBusy: return "DeviceInstallerBusy: Can not start Device's installer";

                // Bluetooth pairing
                case ErrorCode.DevicePairingFailed: return "DevicePairingFailed: Pairing failed";
                case ErrorCode.DevicePasswordWrong: return "DevicePasswordWrong: Wrong password on device";
                case ErrorCode.DevicePasswordInvalid: return "DevicePasswordId: Password includes invalid characters or missing";
                default: return "Undefined errorCode code"; // shouldn't occur
            }
        }

        static ErrorCode m_LastErrorCode;
        public static ErrorCode LastErrorCode
        {
            get { return m_LastErrorCode; }
        }

        public static ErrorCode[] EC_Empty = new ErrorCode[0];
        public static ErrorCode[] EC_OK = new ErrorCode[] { ErrorCode.OK};
        public static ErrorCode[] EC_Cancelled = new ErrorCode[] { ErrorCode.Cancelled };

        public static bool Test(
            ErrorCode errorCode, ErrorCode[] validCodes, ErrorCode[] silentErrorCodes, 
            string customMessage, bool throwExeption)
        {
            // if valid, true
            foreach (ErrorCode e in validCodes) if (e == errorCode) return true;

            // last
            m_LastErrorCode = errorCode;

            string msg = "";
            if (customMessage != null) msg = customMessage + ", ";
            msg += "ErrorCode: " + errorCode.ToString() + ", " + GetMessage(errorCode);

            // no throw NokiaException
            if (!throwExeption) return false;

            // is error but no throw NokiaException
            foreach (ErrorCode e in silentErrorCodes) if (e == errorCode) return false;

            
            throw new NokiaException(errorCode, msg);
        }

        public static bool TestError(ErrorCode errorCode, ErrorCode[] validCodes, ErrorCode[] silentErrorCodes)
        {
            return Test(errorCode, validCodes, silentErrorCodes, null, true);
        }
        public static bool Test(ErrorCode errorCode, ErrorCode[] validCodes, bool throwExeption)
        {
            return Test(errorCode, validCodes, new ErrorCode[0], null, throwExeption);
        }

        public static bool TestError(ErrorCode errorCode)
        {
            return Test(errorCode, new ErrorCode[] { ErrorCode.OK }, new ErrorCode[0] ,null, true);
        }

        public static bool TestError(ErrorCode errorCode, ErrorCode[] validCodes)
        {
            return Test(errorCode, validCodes, new ErrorCode[0], null, true);
        }

        public static bool TestOnly(ErrorCode errorCode)
        {
            return Test(errorCode, new ErrorCode[] { ErrorCode.OK }, new ErrorCode[0],null, false);
        }

        public static bool TestOnly(ErrorCode errorCode, ErrorCode[] validCodes)
        {
            return Test(errorCode, validCodes, new ErrorCode[0], null, false);
        }

    }

    public class NokiaException : Exception
    {
        public NokiaException(ErrorCode errorCode, string message)
            : base(message)
        {
            m_error = errorCode;
        }

        public NokiaException(ErrorCode errorCode)
            : base(ErrorHelper.GetMessage(errorCode))
        {
            m_error = errorCode;
        }

        ErrorCode m_error;

        public ErrorCode Error
        {
            get { return m_error; }
        }

        public string ErrorMessage
        {
            get { return ErrorHelper.GetMessage(m_error); }
        }
    }
}
