using System;
using System.Collections;
using System.Text;

using CONNAPI;

namespace MosTech.Nokia
{

    public enum CopyDirection : uint
    {
        PhoneToPc = 2,
        PcToPhone = 4
    }

    public class FileSystem : IDisposable
    {
        public FileSystem(string serialNumber)
        {
            _serialNumber = serialNumber;
            _FSCallBack = new ConnAPI.FSCallBack(NotifyCallBack);
            uint dwResult = ConnAPI.CONAInitialize(CONAPI_CONSTANTS.CONA_API_VERSION, "Nokia", IntPtr.Zero);

            testExcept(dwResult);
        }

        private unsafe bool OpenFS()
        {
            return OpenFS(Media.All);
        }

        private unsafe bool OpenFS(Media media)
        {
            //if ((_hFSHandle != IntPtr.Zero) && (_isPersistent)) return true;

            uint dwMedia = (uint)media;
            uint dwDeviceID = 0;
            uint dwResult = 0;

            // Open Filesystem
            fixed (IntPtr* pp = &_hFSHandle)
            {
                dwResult = ConnAPI.CONAOpenFS(_serialNumber, &dwMedia, pp, &dwDeviceID);
            }
            testExcept(dwResult);

            // Register callback function
            dwResult = ConnAPI.CONARegisterFSNotifyCallback(_hFSHandle, CONAPI_CONSTANTS.CONAPI_REGISTER, _FSCallBack);
            if (!testResult(dwResult))
            {
                ConnAPI.CONACloseFS(_hFSHandle);
                testExcept(dwResult);
            }

            m_dwMedia = (Media)dwMedia;
            return true;
        }

        private unsafe void CloseFS()
        {
            //  Unregister FS callback function
            uint dwResult = ConnAPI.CONARegisterFSNotifyCallback(_hFSHandle, CONAPI_CONSTANTS.CONAPI_UNREGISTER, _FSCallBack);
            testExcept(dwResult);

            // close
            if (_hFSHandle != IntPtr.Zero)
            {
                dwResult = ConnAPI.CONACloseFS(_hFSHandle);
                _hFSHandle = IntPtr.Zero;
                testExcept(dwResult);
            }
        }


        public unsafe bool DeleteFile(string path)
        {
            _lastOperationInfo = path;
            OpenFS();
            uint dwResult = ConnAPI.CONADeleteFile(_hFSHandle, SplitItem(path), SplitPath(path));
            CloseFS();

            return testExcept(dwResult);
        }
        public unsafe bool DeleteFolder(string path, bool withFiles)
        {
            _lastOperationInfo = path;
            uint delOptions = withFiles ? CONAPI_CONSTANTS.CONA_DELETE_FOLDER_WITH_FILES : CONAPI_CONSTANTS.CONA_DELETE_FOLDER_EMPTY;
            OpenFS();
            Error dwResult = ConnAPI.CONADeleteFolder
                (_hFSHandle, SplitItem(path), delOptions, SplitPath(path));
            CloseFS();
            return testExcept(dwResult);
        }

        public unsafe bool CreateFolder(string path)
        {
            _lastOperationInfo = path;
            OpenFS();

            uint dwResult = ConnAPI.CONACreateFolder(_hFSHandle, SplitItem(path), SplitPath(path));
            CloseFS();

            return testExcept(dwResult);
        }



        public unsafe bool CopyFile(CopyDirection direction, string source, string target,
            bool overwrite, bool renameIfExists)
        {
            _lastOperationInfo = source + ";" + target;

            string strFileName = "";
            string strSourcePath = "";
            string strTargetPath = "";

            if (direction == CopyDirection.PcToPhone)
            {
                strFileName = SplitItem(target);
                strSourcePath = source;
                strTargetPath = SplitPath(target);
            }
            else
            {
                strFileName = SplitItem(source);
                strSourcePath = SplitPath(source);
                strTargetPath = target;
            }

            uint options = (uint)direction;
            if (overwrite) options = options | CONAPI_CONSTANTS.CONA_OVERWRITE;
            if (renameIfExists) options = options | CONAPI_CONSTANTS.CONA_RENAME;

            OpenFS();
            uint dwResult = ConnAPI.CONACopyFile(_hFSHandle, options, strFileName, strSourcePath, strTargetPath);
            CloseFS();


            return ErrorHelper.TestError((ErrorCode)dwResult, ErrorHelper.EC_OK, ErrorHelper.EC_Cancelled);


        }

        public unsafe bool MoveFile(CopyDirection direction, string source, string target,
            bool overwrite)
        {
            _lastOperationInfo = source + ";" + target;
            string strFileName = "";
            string strSourcePath = "";
            string strTargetPath = "";

            if (direction == CopyDirection.PcToPhone)
            {
                strFileName = SplitItem(target);
                strSourcePath = source;
                strTargetPath = SplitPath(target);
            }
            else
            {
                strFileName = SplitItem(source);
                strSourcePath = SplitPath(source);
                strTargetPath = target;
            }

            uint options = (uint)direction;
            if (overwrite) options = options | CONAPI_CONSTANTS.CONA_OVERWRITE;

            OpenFS();
            uint dwResult = ConnAPI.CONAMoveFile(_hFSHandle, options, strFileName, strSourcePath, strTargetPath);
            CloseFS();

            return testExcept(dwResult);
        }


        public unsafe bool RenameFile(string fileName, string newFileName)
            
        {
            _lastOperationInfo = fileName + ";" + newFileName;

            if (SplitPath(fileName) != SplitPath(newFileName))
                throw new ArgumentException("Both files must be in one directory");

           
            OpenFS();
            uint dwResult = ConnAPI.CONARenameFile(_hFSHandle, SplitItem(fileName), SplitItem(newFileName), SplitPath(fileName));
            CloseFS();

            return ErrorHelper.TestError((ErrorCode)dwResult);
        }
        //public unsafe bool FileExists(string path)
        //{
        //    System.Windows.Forms.MessageBox.Show(path);
        //    uint dwResult = 0;
        //    OpenFS();


        //        IntPtr findHandle;
        //        IntPtr* hFind = &findHandle;

        //        dwResult = ConnAPI.CONAFindBegin(_hFSHandle, CONAPI_CONSTANTS.CONA_FIND_USE_CACHE, hFind, SplitPath(path));
        //        CloseFS();
        //        testExcept(dwResult);

        //        return true;

        //}


        public unsafe NokiaInfo[] GetItems(string folder)
        {
            _lastOperationInfo = folder;
            ArrayList preparedFolders = new ArrayList();

            uint dwResult = 0;
            OpenFS();
            try
            {

                IntPtr findHandle;
                IntPtr* hFind = &findHandle;

                dwResult = ConnAPI.CONAFindBegin(_hFSHandle, CONAPI_CONSTANTS.CONA_FIND_USE_CACHE, hFind, folder);



                // --- folders ----

                ConnAPI.CONAPI_FOLDER_INFO FolderInfo;// = new ConnAPI.CONAPI_FOLDER_INFO();

                dwResult = CONAPI_ERRORS.CONA_OK;
                while (dwResult == CONAPI_ERRORS.CONA_OK)
                {
                    dwResult = ConnAPI.CONAFindNextFolder(*hFind, &FolderInfo);
                    if (dwResult == CONAPI_ERRORS.CONA_OK)
                    {
                        String strTemp = new String(FolderInfo.pstrName);
                        preparedFolders.Add(new DirInfo(strTemp));
                        ConnAPI.CONAFreeFolderInfoStructure(&FolderInfo);
                    }
                }

                if (dwResult != CONAPI_ERRORS.ECONA_ALL_LISTED) testExcept(dwResult);


                // --- files ----

                ConnAPI.CONAPI_FILE_INFO FileInfo;// = new ConnAPI.CONAPI_FILE_INFO();

                dwResult = CONAPI_ERRORS.CONA_OK;
                while (dwResult == CONAPI_ERRORS.CONA_OK)
                {
                    dwResult = ConnAPI.CONAFindNextFile(*hFind, &FileInfo);
                    if (dwResult == CONAPI_ERRORS.CONA_OK)
                    {
                        String strTemp = new String(FileInfo.pstrName);
                        preparedFolders.Add(new FileInfo(strTemp, FileInfo.dwFileSize, FileInfo.dwAttributes, FileInfo.tFiletime));
                        ConnAPI.CONAFreeFileInfoStructure(&FileInfo);
                    }
                }
                if (dwResult != CONAPI_ERRORS.ECONA_ALL_LISTED) testExcept(dwResult);
            }
            finally
            {
                CloseFS();
            }


            NokiaInfo[] ni = new NokiaInfo[preparedFolders.Count];

            for (int i = 0; i < preparedFolders.Count; i++) ni[i] = (NokiaInfo)preparedFolders[i];
            return ni;
        }

        public static event FileSystemEventHandler Notify;

        private uint NotifyCallBack(int operation, int status, int transferredBytes, int allBytes)
        {
            short l_status = -1;
            if ((status >= 0) && (status <= 100)) l_status = (short)status;

            FileSystemEventArgs evargs = 
                new FileSystemEventArgs(l_status, (FileSystemOperation)operation, _lastOperationInfo);

            if (Notify != null) Notify(this, evargs);

            // user cancel
            return evargs.Cancel ? (uint)ErrorCode.Cancelled : (uint)ErrorCode.OK;
        }

        #region IDisposable Members

        public void Dispose()
        {
            throw new Exception("The method or operation is not implemented.");
        }

        #endregion

        private bool testExcept(uint result)
        {
            if (result != CONAPI_ERRORS.CONA_OK)
            {
                _lastError = result;
                throw new Exception(CONAPI_ERRORS.CONAError2String(result));
            }
            return true;
        }

        private bool testResult(uint result)
        {
            return (result == CONAPI_ERRORS.CONA_OK);

        }

        private static string SplitPath(string path)
        {
            int sep = path.LastIndexOf('\\');
            return path.Substring(0, sep);
        }

        private static string SplitItem(string path)
        {
            int sep = path.LastIndexOf('\\');
            return path.Substring(sep + 1);
        }




        public uint LastError
        {
            get { return _lastError; }
            set { _lastError = value; }
        }

        private string _lastOperationInfo;
        
        private uint _lastError;
        private bool _isPersistent;

        private string _serialNumber;
        private IntPtr _hFSHandle;
        Media m_dwMedia;

        private ConnAPI.FSCallBack _FSCallBack;
    }

    public class FileSystemEventArgs : EventArgs
    {
        public FileSystemEventArgs(short progress, FileSystemOperation operation, string operationInfo)
        {
            m_progress = progress;
            m_operation = operation;
            m_OperationInfo = operationInfo;
        }
        short m_progress;
        FileSystemOperation m_operation;
        bool m_cancel;
        string m_OperationInfo;

        public string OperationInfo
        {
            get { return m_OperationInfo; }
        }

        public bool Cancel
        {
            get { return m_cancel; }
            set { m_cancel = value; }
        }

        public FileSystemOperation Operation
        {
            get { return m_operation; }
        }

        public short Progress
        {
            get { return m_progress; }
        }
    }

    public delegate void FileSystemEventHandler(object sender, FileSystemEventArgs e);

    public enum FileSystemOperation : int
    {
        None = 0,
        RefreshDeviceMemoryValues = 0x00000001,
        SetCurrentFolder = 0x00000002,
        FindBegin = 0x00000004,
        CreateFolder = 0x00000008,
        DeleteFolder = 0x00000010,
        RenameFolder = 0x00000020,
        GetFileInfo = 0x00000040,
        DeleteFile = 0x00000080,
        MoveFile = 0x00000100,
        CopyFile = 0x00000200,
        RenameFile = 0x00000400,
        ReadFile = 0x00000800,
        WriteFile = 0x00001000,
        ConnectionLost = 0x00002000,
        InstallApplication = 0x00004000
    }
}
