//------------------------------------------------------------------------------
// RAM copy file system plugin v1.3 for Total Commander
//
// Created by Summer, Digital Raptors (c) 2004
//
// e-mail: summr@freemail.hu
//------------------------------------------------------------------------------

library RamCopy;

uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Registry,
  StrUtils,
  ShellAPI,
  CommCtrl,
  Math,
  fsPlugin;

{$E wfx}

{$R *.res}

//------------------------------------------------------------------------------
// Local types, variables
//------------------------------------------------------------------------------

type
  Int64Rec = packed record
    case Integer of
      0: (Int64: Int64);
      1: (Lo, Hi: DWord);
  end;

  PRAMFile = ^TRAMFile;
  TRAMFile = record
    SrcFileName: string;
    DstFileName: string;
    FileSize: Int64;
    BlockOffset: Longword;
    BlockSize: Longword;
    Attributes: DWord;
    LastWriteTime: TFileTime;
    Copied: Int64;
  end;

  TOperation = (opRead, opWrite, opRemoveDir);
  TCopyMoveDir = (cmdNothing, cmdOriginalToVirtual, cmdVirtualToOriginal,
    cmdVirtualToVirtual);

const
  SubKey = 'Software\Digital Raptors\RamCopyPlugin';
  UFRPName = 'UFRPercentage';
  LNGName = 'Language';

  SD_ID = 101; // Settings dialog
  SD_IDOK = 1;
  SD_IDCANCEL = 2;
  SD_IDSLIDER = 3;
  SD_IDPERCENTLABEL = 4;
  SD_IDLNGCOMBO = 5;
  SD_IDTEXTID1 = 6;
  SD_IDTEXTID2 = 7;

  CD_ID = 102; // Copy dialog
  CD_IDRPPROGRESS = 10;
  CD_IDWPPROGRESS = 11;
  CD_IDRFPROGRESS = 12;
  CD_IDRPLABEL = 13;
  CD_IDWPLABEL = 14;
  CD_IDRFLABEL = 15;
  CD_IDSTATUS = 16;
  CD_IDTEXTID5 = 17;
  CD_IDTEXTID6 = 18;
  CD_IDTEXTID7 = 19;

var
  PluginNr: Integer;
  ProgressProc: TProgressProc;
  IsRootDir: Boolean;
  StrBuffer: array [0..255] of Char;
  DriveCounter: Integer;
  CopyMove: Boolean;
  MoveOperation: Boolean;
  UFRPercentage: Single;
  CopyBufferSize: Longword;
  CopyBuffer: PByteArray;
  BufferPtr: Longword;
  BufferPtrTop: Longword;
  RAMFiles: TList;
  LastWriteFHandle: DWord;
  SettingsDialog: HWND;
  CopyDialog: HWND;
  CopyDialogCreated: Boolean;
  LngTexts: TStringList;
  LngFiles: TStringList;
  LngFileIdx: Integer;
  PreFileOperations: Boolean;
  FirstDirCache: TStringList;
  DirCache: TStringList;
  CopyMoveDir: TCopyMoveDir;
  MoveFromPath: string;
  MoveToPath: string;

//------------------------------------------------------------------------------
// DLL entry point
//------------------------------------------------------------------------------

procedure WFXMain (Reason: Integer);
begin
  case Reason of
    DLL_PROCESS_ATTACH:
    begin
      LngTexts := TStringList.Create;
      LngFiles := TStringList.Create;
    end;
    DLL_PROCESS_DETACH:
    begin
      LngFiles.Free;
      LngTexts.Free;
    end;
  end;
end;

//------------------------------------------------------------------------------
// Language functions
//------------------------------------------------------------------------------

procedure ExploreLngFiles;
var
  SearchRec: TSearchRec;
  WFXName: array [0..MAX_PATH] of Char;
  WFXDir: string;
begin
  GetModuleFileName (hInstance, @WFXName, MAX_PATH);
  WFXDir := ExtractFilePath (WFXName);

  LngFiles.Clear;
  if FindFirst (WFXDir + '*.rcl', faArchive or faReadOnly, SearchRec) = 0 then
  begin
    repeat
      LngFiles.Add (WFXDir + SearchRec.Name);
    until FindNext (SearchRec) <> 0;
    FindClose (SearchRec);
  end;
end;

procedure LoadLngTexts;
var
  Loaded: Boolean;
begin
  LngTexts.Clear;

  Loaded := False;
  if LngFileIdx > 0 then
    if FileExists (LngFiles[LngFileIdx - 1]) then
    begin
      LngTexts.LoadFromFile (LngFiles[LngFileIdx - 1]);
      Loaded := True;
    end;

  if not Loaded then
  begin
    LngTexts.Add ('RamCopy settings');                            // 0
    LngTexts.Add ('Use free physical RAM for copy or move:');     // 1
    LngTexts.Add ('Language file:');                              // 2
    LngTexts.Add ('&Ok');                                         // 3
    LngTexts.Add ('&Cancel');                                     // 4
    LngTexts.Add ('Reading process:');                            // 5
    LngTexts.Add ('Writing process:');                            // 6
    LngTexts.Add ('RAM fullness:');                               // 7
    LngTexts.Add ('Reading file:');                               // 8
    LngTexts.Add ('Writing file:');                               // 9
    LngTexts.Add ('Remove directory:');                           // 10
    LngTexts.Add ('Cannot read the source file!');                // 11
    LngTexts.Add ('Cannot write the destination file!');          // 12
    LngTexts.Add ('Cannot copy or move file(s) to itself!');      // 13
    LngTexts.Add ('Cannot access to the drive content!');         // 14
    LngTexts.Add ('Could not allocate RAM!');                     // 15
    LngTexts.Add ('Settings');                                    // 16
  end;
end;

function GetLngText (Index: Integer): PChar;
begin
  try
    Result := PChar (LngTexts[Index]);
  except
    on E: Exception do
      Result := '';
  end;
end;

procedure UseLngOnSettingsDialog;
begin
  SetWindowText (SettingsDialog, GetLngText (0));
  SetDlgItemText (SettingsDialog, SD_IDTEXTID1, GetLngText (1));
  SetDlgItemText (SettingsDialog, SD_IDTEXTID2, GetLngText (2));
  SetDlgItemText (SettingsDialog, SD_IDOK, GetLngText (3));
  SetDlgItemText (SettingsDialog, SD_IDCANCEL, GetLngText (4));
end;

procedure UseLngOnCopyDialog;
begin
  SetDlgItemText (CopyDialog, CD_IDTEXTID5, GetLngText (5));
  SetDlgItemText (CopyDialog, CD_IDTEXTID6, GetLngText (6));
  SetDlgItemText (CopyDialog, CD_IDTEXTID7, PChar (GetLngText (7) + Format (
    ' [%f MB]', [CopyBufferSize / (1024 * 1024)])));
end;

//------------------------------------------------------------------------------
// Copy dialog functions
//------------------------------------------------------------------------------

procedure UpdateCopyDialogReadProgress (FileSize, CopiedSize: Int64);
var
  Percent: Double;
begin
  Percent := CopiedSize / FileSize * 100.0;
  SendDlgItemMessage (CopyDialog, CD_IDRPPROGRESS, PBM_SETPOS, Round (
    Percent), 0);
  SetDlgItemText (CopyDialog, CD_IDRPLABEL, PChar (Format ('%f%%', [Percent])));
end;

procedure UpdateCopyDialogWriteProgress (FileSize, CopiedSize: Int64);
var
  Percent: Double;
begin
  Percent := CopiedSize / FileSize * 100.0;
  SendDlgItemMessage (CopyDialog, CD_IDWPPROGRESS, PBM_SETPOS, Round (
    Percent), 0);
  SetDlgItemText (CopyDialog, CD_IDWPLABEL, PChar (Format ('%f%%', [Percent])));
end;

procedure UpdateCopyDialogRAMProgress (Offset: Longword);
var
  Percent: Double;
begin
  Percent := Offset / CopyBufferSize * 100.0;
  SendDlgItemMessage (CopyDialog, CD_IDRFPROGRESS, PBM_SETPOS, Round (
    Percent), 0);
  SetDlgItemText (CopyDialog, CD_IDRFLABEL, PChar (Format ('%f%%', [Percent])));
end;

procedure UpdateCopyDialogStatus (Operation: TOperation; FileName: string);
begin
  FileName := ExtractFileName (FileName);
  case Operation of
    opRead:
      SetDlgItemText (CopyDialog, CD_IDSTATUS, PChar (GetLngText (8) + ' ' +
        FileName));
    opWrite:
      SetDlgItemText (CopyDialog, CD_IDSTATUS, PChar (GetLngText (9) + ' ' +
        FileName));
    opRemoveDir:
      SetDlgItemText (CopyDialog, CD_IDSTATUS, PChar (GetLngText (10) + ' ' +
        FileName));
  end;
end;

function CopyDialogProc (hwndDlg: HWND; uMsg: UINT; wParam: WPARAM; lParam:
  LPARAM): Boolean; stdcall;
var
  dRect: TRect;
  oRect: TRect;
begin
  Result := False;
  case uMsg of
    WM_INITDIALOG:
    begin
      GetWindowRect (hwndDlg, dRect);
      GetWindowRect (GetParent (hwndDlg), oRect);
      SetWindowPos (hwndDlg, HWND_TOP, oRect.Left + ((oRect.Right -
        oRect.Left) - (dRect.Right - dRect.Left)) div 2, oRect.Bottom, 0, 0,
        SWP_NOSIZE);
      Result := True;
    end;
  end;
end;

procedure CreateCopyDialog;
begin
  if CopyDialogCreated then
    Exit;
  CopyDialog := CreateDialog (HInstance, MAKEINTRESOURCE (CD_ID),
    GetForegroundWindow, @CopyDialogProc);
  UseLngOnCopyDialog;
  ShowWindow (CopyDialog, SW_SHOWNORMAL);
  UpdateWindow (CopyDialog);
  CopyDialogCreated := True;
end;

//------------------------------------------------------------------------------
// RAM functions
//------------------------------------------------------------------------------

function InitRAMCopy: Boolean;
var
  MS: TMemoryStatus;
begin
  MS.dwLength := SizeOf (MS);
  GlobalMemoryStatus (MS);
  CopyBufferSize := Round (MS.dwAvailPhys * UFRPercentage);
  if CopyBufferSize > 0 then
    CopyBuffer := Pointer (GlobalAlloc (GMEM_FIXED, CopyBufferSize));
  BufferPtr := 0;
  BufferPtrTop := 0;
  Result := CopyBuffer <> nil;
  if Result then
  begin
    RAMFiles := TList.Create;
    Result := RAMFiles <> nil;
  end;
end;

procedure DoneRAMCopy;
begin
  if CopyDialogCreated then
  begin
    DestroyWindow (CopyDialog);
    CopyDialogCreated := False;
  end;
  if CopyBuffer <> nil then
  begin
    if RAMFiles <> nil then
    begin
      RAMFiles.Free;
      RAMFiles := nil;
    end;
    GlobalFree (Cardinal (CopyBuffer));
    CopyBuffer := nil;
  end;
end;

procedure ClearRAMFilesItems;
var
  I: Integer;
begin
  for I := 0 to RAMFiles.Count - 1 do
    if RAMFiles.Items[I] <> nil then
      Finalize (PRAMFile (RAMFiles.Items[I])^);
  RAMFiles.Clear;
end;

function ReadFileToRAM (SrcHandle: THandle; RAMFile: PRAMFile): Integer;
var
  Reading: Longword;
  ReadBytes: Longword;
  Size: Longword;
  Offset: Longword;
begin
  UpdateCopyDialogStatus (opRead, RAMFile^.SrcFileName);

  Size := RAMFile^.BlockSize;
  Offset := RAMFile^.BlockOffset;
  while Size > 0 do
  begin
    Reading := Min (Size, 65536);
    if not ReadFile (SrcHandle, CopyBuffer[Offset], Reading, ReadBytes,
      nil) then
    begin
      if MessageBox (0, GetLngText (11), 'RamCopy', MB_OKCANCEL or
        MB_ICONERROR or MB_TASKMODAL) = IDCANCEL then
        Result := FS_FILE_USERABORT
      else
        Result := FS_FILE_READERROR;
      Exit;
    end;
    Inc (Offset, ReadBytes);
    Inc (RAMFile^.Copied, ReadBytes);
    if ProgressProc (PluginNr, PChar (RAMFile^.SrcFileName), PChar ('RAM'),
      Round ((1.0 - Size / RAMFile^.BlockSize) * 100)) = 1 then
    begin
      Result := FS_FILE_USERABORT;
      Exit;
    end;
    Dec (Size, ReadBytes);
    UpdateCopyDialogReadProgress (RAMFile^.FileSize, RAMFile^.Copied);
    UpdateCopyDialogRAMProgress (Offset);
  end;
  Result := FS_FILE_OK;
end;

function WriteFileFromRAM (RAMFile: PRAMFile; CloseFile: Boolean): Integer;
var
  DstHandle: THandle;
  Writing: Longword;
  WriteBytes: Longword;
  BlockSize: Longword;
begin
  UpdateCopyDialogStatus (opWrite, RAMFile^.DstFileName);

  Result := FS_FILE_OK;
  if LastWriteFHandle = INVALID_HANDLE_VALUE then
    DstHandle := CreateFile (PChar (RAMFile^.DstFileName), GENERIC_WRITE,
      0, nil, CREATE_ALWAYS, RAMFile^.Attributes, 0)
  else
    DstHandle := LastWriteFHandle;
  if DstHandle <> INVALID_HANDLE_VALUE then
  begin
    BlockSize := RAMFile^.BlockSize;
    while RAMFile^.BlockSize > 0 do
    begin
      Writing := Min (RAMFile^.BlockSize, 65536);
      if not WriteFile (DstHandle, CopyBuffer[RAMFile^.BlockOffset],
        Writing, WriteBytes, nil) then
      begin
        Result := FS_FILE_WRITEERROR;
        CloseFile := True;
        Break;
      end;
      Inc (RAMFile^.BlockOffset, WriteBytes);
      if ProgressProc (PluginNr, PChar ('RAM'), PChar (RAMFile^.DstFileName),
        Round ((1.0 - RAMFile^.BlockSize / BlockSize) * 100)) = 1 then
      begin
        Result := FS_FILE_USERABORT;
        CloseFile := True;
        Break;
      end;
      Dec (RAMFile^.BlockSize, WriteBytes);
      UpdateCopyDialogWriteProgress (RAMFile^.FileSize, RAMFile^.Copied -
        RAMFile^.BlockSize);
      UpdateCopyDialogRAMProgress (BufferPtrTop - RAMFile^.BlockOffset);
    end;
    if CloseFile then
    begin
      SetFileTime (DstHandle, @RAMFile^.LastWriteTime, @RAMFile^.LastWriteTime,
        @RAMFile^.LastWriteTime);
      CloseHandle (DstHandle);
      LastWriteFHandle := INVALID_HANDLE_VALUE;
    end
    else
    begin
      FlushFileBuffers (DstHandle);
      LastWriteFHandle := DstHandle;
    end;
  end
  else
    Result := FS_FILE_WRITEERROR;

  if Result = FS_FILE_WRITEERROR then
    if MessageBox (0, GetLngText (12), 'RamCopy', MB_OKCANCEL or MB_ICONERROR or
      MB_TASKMODAL) = IDCANCEL then
        Result := FS_FILE_USERABORT;
end;

function FlushRAM (LastIOPending: Boolean): Integer;
var
  I: Integer;
  RAMFile: PRAMFile;
  CloseFile: Boolean;
begin
  for I := 0 to RAMFiles.Count - 1 do
  begin
    RAMFile := PRAMFile (RAMFiles.Items[I]);
    CloseFile := not ((I = RAMFiles.Count - 1) and LastIOPending);
    Result := WriteFileFromRAM (RAMFile, CloseFile);
    if Result <> FS_FILE_OK then
      Exit;
    if CloseFile then
    begin
      if MoveOperation then
        Windows.DeleteFile (PChar (RAMFile^.SrcFileName));
      Finalize (RAMFile^);
      RAMFiles.Items[I] := nil;
    end;
  end;

  RAMFiles.Clear;
  BufferPtr := 0;
  Result := FS_FILE_OK;
end;

function CopyThroughRAM (SrcFileName, DstFileName: string): Integer;
var
  RAMFile: PRAMFile;
  SrcHandle: THandle;
  FileSize: Int64Rec;
  IOPending: Boolean;
begin
  if MoveOperation and (LowerCase (SrcFileName[1]) = LowerCase (
    DstFileName[1])) then
  begin
    MoveFileEx (PChar (SrcFileName), PChar (DstFileName),
      MOVEFILE_REPLACE_EXISTING);
    Result := FS_FILE_OK;
    Exit;
  end;

  SrcHandle := CreateFile (PChar (SrcFileName), GENERIC_READ, FILE_SHARE_READ,
    nil, OPEN_EXISTING, 0, 0);
  if SrcHandle = INVALID_HANDLE_VALUE then
  begin
    Result := FS_FILE_NOTFOUND;
    Exit;
  end;

  FileSize.Lo := GetFileSize (SrcHandle, @FileSize.Hi);
  if FileSize.Lo = $FFFFFFFF then
    if GetLastError <> NO_ERROR then
    begin
      CloseHandle (SrcHandle);
      Result := FS_FILE_READERROR;
      Exit;
    end;

  New (RAMFile);
  RAMFile^.SrcFileName := SrcFileName;
  RAMFile^.DstFileName := DstFileName;
  RAMFile^.FileSize := FileSize.Int64;
  RAMFile^.BlockOffset := BufferPtr;
  if BufferPtr + FileSize.Int64 > CopyBufferSize then
    RAMFile^.BlockSize := CopyBufferSize - BufferPtr
  else
    RAMFile^.BlockSize := FileSize.Int64;
  RAMFile^.Attributes := GetFileAttributes (PChar (SrcFileName));
  if RAMFile^.Attributes = $FFFFFFFF then
    RAMFile^.Attributes := FILE_ATTRIBUTE_NORMAL or FILE_ATTRIBUTE_ARCHIVE;
  ZeroMemory (@RAMFile^.LastWriteTime, SizeOf (TFileTime));
  GetFileTime (SrcHandle, nil, nil, @RAMFile^.LastWriteTime);
  RAMFile^.Copied := 0;

  repeat
    Result := ReadFileToRAM (SrcHandle, RAMFile);

    if Result <> FS_FILE_OK then
    begin
      Finalize (RAMFile^);
      if Result = FS_FILE_USERABORT then
        ClearRAMFilesItems;
      CloseHandle (SrcHandle);
      if LastWriteFHandle <> INVALID_HANDLE_VALUE then
        CloseHandle (LastWriteFHandle);
      Exit;
    end;

    RAMFiles.Add (RAMFile);
    Inc (BufferPtr, RAMFile^.BlockSize);
    BufferPtrTop := BufferPtr;
    IOPending := RAMFile^.Copied < FileSize.Int64;

    if BufferPtr = CopyBufferSize then
    begin
      Result := FlushRAM (IOPending);
      if  Result <> FS_FILE_OK then
      begin
        ClearRAMFilesItems;
        CloseHandle (SrcHandle);
        if LastWriteFHandle <> INVALID_HANDLE_VALUE then
          CloseHandle (LastWriteFHandle);
        Exit;
      end;
    end;

    if IOPending then
    begin
      RAMFile^.BlockOffset := BufferPtr;
      if BufferPtr + (FileSize.Int64 - RAMFile^.Copied) > CopyBufferSize then
        RAMFile^.BlockSize := CopyBufferSize - BufferPtr
      else
        RAMFile^.BlockSize := FileSize.Int64 - RAMFile^.Copied;
    end;
  until not IOPending;

  CloseHandle (SrcHandle);

  Result := FS_FILE_OK;
end;

//------------------------------------------------------------------------------
// Local functions
//------------------------------------------------------------------------------

function GetDriveName (DriveNr: Integer): string;
var
  Drives: DWord;
  BitPos: Integer;
  RootPath: string;
  DriveInfoStr: string;
  Unused: Cardinal;
begin
  Result := '';
  Drives := GetLogicalDrives;
  BitPos := 0;
  while BitPos < 32 do
  begin
    if (Drives and (1 shl BitPos)) > 0 then
    begin
      Dec (DriveNr);
      if DriveNr < 0 then
      begin
        Result := Chr (Ord ('A') + BitPos);
        RootPath := Result + ':\';
        case GetDriveType (PChar (RootPath)) of
          DRIVE_REMOVABLE:
            if (Result = 'A') or (Result = 'B') then
              DriveInfoStr := '3 1/2"'
            else
              DriveInfoStr := 'REMOVABLE';
          DRIVE_FIXED:
          begin
            GetVolumeInformation (PChar (Result + ':\'), StrBuffer, 255, nil,
              Unused, Unused, nil, 0);
            DriveInfoStr := StrBuffer;
          end;
          DRIVE_CDROM:
            DriveInfoStr := 'CD-ROM';
          DRIVE_RAMDISK:
            DriveInfoStr := 'RAMDISK';
          else
            DriveInfoStr := 'UNKNOWN';
        end;
        Result := Result + ': (' + DriveInfoStr + ')';
        Break;
      end
    end;
    Inc (BitPos);
  end;
  if DriveNr = 0 then
    Result := GetLngText (16) + ' ...';
end;

function GetPath (Path: PChar): string;
var
  BSPos: Integer;
begin
  Result := Path + 1;
  BSPos := Pos ('\', Result);
  if BSPos = 0 then
    Delete (Result, 3, Length (Result) - 2)
  else
    Delete (Result, 3, BSPos - 3);
end;

procedure EmulateDriveForWin32FindData (var Win32FindData: TWin32FindData;
  DriveName: string);
begin
  ZeroMemory (@Win32FindData, SizeOf (TWin32FindData));
  if DriveName = GetLngText (16) + ' ...' then
  begin
    Win32FindData.dwFileAttributes := FILE_ATTRIBUTE_NORMAL;
    Win32FindData.nFileSizeLow := $FFFFFFFE;
  end
  else
    Win32FindData.dwFileAttributes := FILE_ATTRIBUTE_DIRECTORY;
  Win32FindData.ftLastWriteTime.dwLowDateTime := $FFFFFFFE;
  Win32FindData.ftLastWriteTime.dwHighDateTime := $FFFFFFFF;
  StrPLCopy (Win32FindData.cFileName, DriveName, MAX_PATH - 1);
end;

function CheckPathDifference (SrcPath, DstPath: string): Boolean;
begin
  Result := True;
  if LowerCase (SrcPath) = LowerCase (DstPath) then
  begin
    MessageBox (0, GetLngText (13), 'RamCopy', MB_OK or MB_ICONERROR or
      MB_TASKMODAL);
    Result := False;
  end;
end;

procedure ReadSettings;
var
  Reg: TRegistry;
begin
  Reg := TRegistry.Create;
  UFRPercentage := 0.75;
  LngFileIdx := 0;
  if Reg.OpenKey (SubKey, True) then
  begin
    if Reg.ValueExists (UFRPName) then
      UFRPercentage := Reg.ReadFloat (UFRPName)
    else
      Reg.WriteFloat (UFRPName, UFRPercentage);
    if Reg.ValueExists (LNGName) then
      LngFileIdx := LngFiles.IndexOf (Reg.ReadString (LNGName)) + 1;
    Reg.CloseKey;
  end;
  Reg.Free;
end;

procedure WriteSettings;
var
  Reg: TRegistry;
begin
  Reg := TRegistry.Create;
  if Reg.OpenKey (SubKey, True) then
  begin
    Reg.WriteFloat (UFRPName, UFRPercentage);
    if LngFileIdx = 0 then
      Reg.DeleteValue (LNGName)
    else
      Reg.WriteString (LNGName, LngFiles[LngFileIdx - 1]);
    Reg.CloseKey;
  end;
  Reg.Free;
end;

function SettingsDialogProc (hwndDlg: HWND; uMsg: UINT; wParam: WPARAM; lParam:
  LPARAM): Boolean; stdcall;
var
  dRect: TRect;
  oRect: TRect;
  UFRP: Integer;
  I: Integer;
  LngNames: array of string;
begin
  Result := False;
  case uMsg of
    WM_INITDIALOG:
    begin
      SettingsDialog := hwndDlg;
      GetWindowRect (hwndDlg, dRect);
      GetWindowRect (GetParent (hwndDlg), oRect);
      SetWindowPos (hwndDlg, HWND_TOP, oRect.Left + ((oRect.Right -
        oRect.Left) - (dRect.Right - dRect.Left)) div 2, oRect.Top + ((
        oRect.Bottom - oRect.Top) - (dRect.Bottom - dRect.Top)) div 2, 0, 0,
        SWP_NOSIZE);
      SendDlgItemMessage (hwndDlg, SD_IDSLIDER, TBM_SETRANGE, Integer (False),
        Integer (MAKELONG (10, 100)));
      SendDlgItemMessage (hwndDlg, SD_IDSLIDER, TBM_SETPOS, Integer (True),
        Round (UFRPercentage * 100));
      SendDlgItemMessage (hwndDlg, SD_IDSLIDER, TBM_SETTICFREQ, 5, 0);
      UseLngOnSettingsDialog;
      SendDlgItemMessage (hwndDlg, SD_IDLNGCOMBO, CB_RESETCONTENT, 0, 0);
      SendDlgItemMessage (hwndDlg, SD_IDLNGCOMBO, CB_ADDSTRING, 0,
        Integer (PChar ('English (Default)')));
      SetLength (LngNames, LngFiles.Count);
      for I := 0 to LngFiles.Count - 1 do
      begin
        LngNames[I] := ExtractFileName (LngFiles[I]);
        LngNames[I] := LeftStr (LngNames[I], Length (LngNames[I]) - 4);
        SendDlgItemMessage (hwndDlg, SD_IDLNGCOMBO, CB_ADDSTRING, 0,
          Integer (PChar (LngNames[I])));
      end;
      SetLength (LngNames, 0);
      SendDlgItemMessage (hwndDlg, SD_IDLNGCOMBO, CB_SETCURSEL, LngFileIdx, 0);
      Result := True;
    end;
    WM_COMMAND:
      case LOWORD (wParam) of
        SD_IDOK, SD_IDCANCEL:
        begin
          if LOWORD (wParam) = SD_IDOK then
            UFRPercentage := SendDlgItemMessage (hwndDlg, SD_IDSLIDER,
              TBM_GETPOS, 0, 0) / 100.0;
          EndDialog (hwndDlg, wParam);
          Result := True;
        end;
        SD_IDLNGCOMBO:
          if HIWORD (wParam) = CBN_SELCHANGE then
          begin
            LngFileIdx := SendDlgItemMessage (hwndDlg, SD_IDLNGCOMBO,
              CB_GETCURSEL, 0, 0);
            LoadLngTexts;
            UseLngOnSettingsDialog;
            Result := True;
          end;
      end;
    WM_NOTIFY:
      if wParam = SD_IDSLIDER then
      begin
        UFRP := SendDlgItemMessage (hwndDlg, SD_IDSLIDER, TBM_GETPOS, 0, 0);
        SetDlgItemText (hwndDlg, SD_IDPERCENTLABEL, PChar (IntToStr (UFRP) +
          '%'));
        Result := True;
      end;
  end;
end;

procedure SetMovingPaths (MoveFrom, MoveTo: string);
var
  FI, TI, FD, TD: Integer;
begin
  MoveFrom := LowerCase (MoveFrom);
  MoveTo := LowerCase (MoveTo);
  FI := Length (MoveFrom);
  TI := Length (MoveTo);
  while (FI > 0) and (TI > 0) do
  begin
    FD := FI;
    while (FD > 0) and (MoveFrom[FD] <> '\') do
      Dec (FD);
    TD := TI;
    while (TD > 0) and (MoveTo[TD] <> '\') do
      Dec (TD);
    if MidStr (MoveFrom, FD + 1, FI - FD + 1) <> MidStr (MoveTo, TD + 1,
      TI - TD + 1) then
      Break;
    FI := FD - 1;
    TI := TD - 1;
  end;
  MoveFromPath := LeftStr (MoveFrom, FI);
  MoveToPath := LeftStr (MoveTo, TI);
end;

procedure RemoveCachedDirs;
var
  I: Integer;
  FirstDirName: string;
begin
  for I := 0 to DirCache.Count - 1 do
  begin
    UpdateCopyDialogStatus (opRemoveDir, DirCache[I]);
    Windows.RemoveDirectory (PChar (DirCache[I]));
  end;
  for I := 0 to FirstDirCache.Count - 1 do
  begin
    FirstDirName := MoveFromPath + RightStr (FirstDirCache[I], Length (
      FirstDirCache[I]) - Length (MoveToPath));
    UpdateCopyDialogStatus (opRemoveDir, FirstDirName);
    Windows.RemoveDirectory (PChar (FirstDirName));
  end;
end;

//------------------------------------------------------------------------------
// Exported functions
//------------------------------------------------------------------------------

procedure FsGetDefRootName (DefRootName: PChar; MaxLen: Integer); stdcall;
begin
  StrPCopy (DefRootName, 'RamCopy');
end;

function FsFindFirst (Path: PChar; var Win32FindData: TWin32FindData):
  THandle; stdcall;
var
  FsPath: string;
  DriveName: string;
  ErrorNum: DWord;
begin
  FsPath := GetPath (Path);
  if CopyMove then
  begin
    if CopyBuffer = nil then
    begin
      SetLastError (ERROR_NO_MORE_FILES);
      Result := INVALID_HANDLE_VALUE;
      Exit;
    end;
    IsRootDir := FsPath[Length (FsPath)] = ':';
    if IsRootDir then
    begin
      Result := INVALID_HANDLE_VALUE;
      Exit;
    end;
  end
  else
    IsRootDir := Path = '\';
  if IsRootDir then
  begin
    DriveName := GetDriveName (0);
    if DriveName = '' then
      Result := INVALID_HANDLE_VALUE
    else
    begin
      DriveCounter := 0;
      EmulateDriveForWin32FindData (Win32FindData, DriveName);
      Result := INVALID_HANDLE_VALUE - 1;
    end;
  end
  else
  begin
    Result := FindFirstFile (PChar (FsPath + '\*.*'), Win32FindData);
    if Result = INVALID_HANDLE_VALUE then
    begin
      ErrorNum := GetLastError;
      if ErrorNum = ERROR_NOT_READY then
        MessageBox (0, GetLngText (14), 'RamCopy', MB_OK or MB_ICONERROR or
          MB_TASKMODAL);
      if ErrorNum = ERROR_FILE_NOT_FOUND then
        SetLastError (ERROR_NO_MORE_FILES);
    end;
  end;
end;

function FsFindNext (Handle: THandle; var Win32FindData: TWin32FindData):
  Boolean; stdcall;
var
  DriveName: string;
begin
  if IsRootDir then
  begin
    Inc (DriveCounter);
    DriveName := GetDriveName (DriveCounter);
    Result := DriveName <> '';
    if Result then
      EmulateDriveForWin32FindData (Win32FindData, DriveName);
  end
  else
    Result := FindNextFile (Handle, Win32FindData);
end;

function FsFindClose (Handle: THandle): Integer; stdcall;
begin
  if IsRootDir then
    Result := Integer (True)
  else
    Result := Integer (Windows.FindClose (Handle));
end;

function FsInit (PNr: Integer; PProc: TProgressProc; LProc: TLogProc;
  RProc: TRequestProc): Integer; stdcall;
begin
  PluginNr := PNr;
  ProgressProc := PProc;
  CopyMove := False;
  CopyBuffer := nil;
  RAMFiles := nil;
  LastWriteFHandle := INVALID_HANDLE_VALUE;
  CopyDialogCreated := False;
  FirstDirCache := nil;
  DirCache := nil;
  CopyMoveDir := cmdNothing;
  ExploreLngFiles;
  ReadSettings;
  LoadLngTexts;
  Result := 0;
end;

function FsGetFile (RemoteName, LocalName: PChar; CopyFlags: Integer; var RI:
  TRemoteInfo): Integer; stdcall;
var
  RemoteNameStr: string;
begin
  PreFileOperations := False;
  if not CopyMove then
  begin
    Result := FS_FILE_USERABORT;
    Exit;
  end;
  CreateCopyDialog;
  RemoteNameStr := GetPath (RemoteName);
  MoveOperation := (CopyFlags and FS_COPYFLAGS_MOVE) > 0;
  if CheckPathDifference (RemoteNameStr, LocalName) then
  begin
    if FileExists (LocalName) and ((CopyFlags = 0) or (CopyFlags =
      FS_COPYFLAGS_MOVE)) then
      Result := FS_FILE_EXISTS
    else
      Result := CopyThroughRAM (RemoteNameStr, LocalName);
  end
  else
    Result := FS_FILE_USERABORT;
end;

function FsPutFile (LocalName, RemoteName: PChar; CopyFlags: Integer):
  Integer; stdcall;
var
  RemoteNameStr: string;
begin
  PreFileOperations := False;
  if not CopyMove then
  begin
    Result := FS_FILE_USERABORT;
    Exit;
  end;
  CreateCopyDialog;
  RemoteNameStr := GetPath (RemoteName);
  MoveOperation := (CopyFlags and FS_COPYFLAGS_MOVE) > 0;
  if CheckPathDifference (LocalName, RemoteNameStr) then
  begin
    if FileExists (RemoteNameStr) and ((CopyFlags and FS_COPYFLAGS_OVERWRITE) =
      0) then
      Result := FS_FILE_EXISTS
    else
    begin
      if MoveFromPath = '' then
        SetMovingPaths (LocalName, RemoteNameStr);
      Result := CopyThroughRAM (LocalName, RemoteNameStr);
    end;
  end
  else
    Result := FS_FILE_USERABORT;
end;

function FsRenMovFile (OldName, NewName: PChar; Move, OverWrite: Boolean;
  var RI: TRemoteInfo): Integer; stdcall;
var
  OldNameStr: string;
  NewNameStr: string;
begin
  PreFileOperations := False;
  if not CopyMove then
  begin
    Result := FS_FILE_USERABORT;
    Exit;
  end;
  CreateCopyDialog;
  OldNameStr := GetPath (OldName);
  NewNameStr := GetPath (NewName);
  MoveOperation := Move;
  if CheckPathDifference (OldNameStr, NewNameStr) then
  begin
    if FileExists (NewNameStr) and (not OverWrite) then
      Result := FS_FILE_EXISTS
    else
      Result := CopyThroughRAM (OldNameStr, NewNameStr);
  end
  else
    Result := FS_FILE_USERABORT;
end;

function FsMkDir (Path: PChar): Boolean; stdcall;
var
  DirName: string;
begin
  if not IsRootDir then
  begin
    if (not CopyMove) and (CopyMoveDir <> cmdNothing) then
    begin
      Result := True;
      Exit;
    end;
    DirName := GetPath (Path);
    Result := CreateDirectory (PChar (DirName), nil);
    if Result and CopyMove and PreFileOperations and (CopyMoveDir =
      cmdOriginalToVirtual) then
      FirstDirCache.Insert (0, DirName);
    if Result and CopyMove and MoveOperation and (CopyMoveDir =
      cmdOriginalToVirtual) then
      DirCache.Insert (0, MoveFromPath + RightStr (DirName, Length (DirName) -
        Length (MoveToPath)));
  end
  else
    Result := False;
end;

function FsRemoveDir (RemoteName: PChar): Boolean; stdcall;
var
  DirName: string;
begin
  if not IsRootDir then
  begin
    if (not CopyMove) and (CopyMoveDir <> cmdNothing) then
    begin
      Result := True;
      Exit;
    end;
    DirName := GetPath (RemoteName);
    if CopyMove and MoveOperation and (CopyMoveDir in [cmdVirtualToOriginal,
      cmdVirtualToVirtual]) then
    begin
      DirCache.Add (DirName);
      Result := True;
    end
    else
      Result := RemoveDirectory (PChar (DirName));
  end
  else
    Result := False;
end;

function FsDeleteFile (RemoteName: PChar): Boolean; stdcall;
begin
  if not IsRootDir then
    Result := Windows.DeleteFile (PChar (GetPath (RemoteName)))
  else
    Result := False;
end;

function FsSetAttr (RemoteName: PChar; NewAttr: Integer): Boolean; stdcall;
var
  FileName: string;
  Attr: Integer;
begin
  if not IsRootDir then
  begin
    FileName := GetPath (RemoteName);
    Attr := GetFileAttributes (PChar (FileName));
    Attr := Attr and not (FILE_ATTRIBUTE_READONLY	or FILE_ATTRIBUTE_HIDDEN or
      FILE_ATTRIBUTE_SYSTEM or FILE_ATTRIBUTE_ARCHIVE);
    Attr := Attr or NewAttr;
    Result := SetFileAttributes (PChar (FileName), Attr);
  end
  else
    Result := False;
end;

function FsSetTime (RemoteName: PChar; CreationTime, LastAccessTime,
  LastWriteTime: PFILETIME): Boolean; stdcall;
var
  FHandle: THandle;
begin
  if not IsRootDir then
  begin
    Result := False;
    FHandle := CreateFile (PChar (GetPath (RemoteName)), GENERIC_WRITE, 0, nil,
      OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, 0);
    if FHandle <> INVALID_HANDLE_VALUE then
    begin
      Result := SetFileTime (FHandle, CreationTime, LastAccessTime,
        LastWriteTime);
      CloseHandle (FHandle);
    end;
  end
  else
    Result := False;
end;

function FsExecuteFile (MainWin: HWND; RemoteName: PChar; Verb: PChar): Integer;
  stdcall;
var
  ExecName: string;
begin
  if ((RemoteName = '\') and (Verb = 'properties')) or
    ((RemoteName = '\' + GetLngText (16) + ' ...') and (Verb = 'open')) then
  begin
    if DialogBox (HInstance, MAKEINTRESOURCE (SD_ID), MainWin,
      @SettingsDialogProc) = SD_IDOK then
      WriteSettings;
    Result := FS_EXEC_OK;
  end
  else
  begin
    Result := FS_EXEC_ERROR;
    ExecName := GetPath (RemoteName);
    if FileExists (ExecName) then
      if ShellExecute (MainWin, 'open', PChar (ExecName), nil, PChar (
        ExtractFilePath (ExecName)), SW_SHOWDEFAULT) > 32 then
        Result := FS_EXEC_OK;
  end;
end;

procedure FsStatusInfo (RemoteDir: PChar; InfoStartEnd, InfoOperation:
  Integer); stdcall;
begin
  if InfoStartEnd = FS_STATUS_START then
    if (InfoOperation >= FS_STATUS_OP_GET_SINGLE) and (InfoOperation <=
      FS_STATUS_OP_RENMOV_MULTI) then
    begin
      CopyMove := InitRAMCopy;
      if CopyMove then
      begin
        FirstDirCache := TStringList.Create;
        CopyMove := FirstDirCache <> nil;
        if CopyMove then
        begin
          DirCache := TStringList.Create;
          CopyMove := DirCache <> nil;
        end;
      end;
      case InfoOperation of
        FS_STATUS_OP_GET_SINGLE, FS_STATUS_OP_GET_MULTI:
          CopyMoveDir := cmdVirtualToOriginal;
        FS_STATUS_OP_PUT_SINGLE, FS_STATUS_OP_PUT_MULTI:
          CopyMoveDir := cmdOriginalToVirtual;
        FS_STATUS_OP_RENMOV_SINGLE, FS_STATUS_OP_RENMOV_MULTI:
          CopyMoveDir := cmdVirtualToVirtual;
      end;
      PreFileOperations := True;
      MoveOperation := False;
      MoveFromPath := '';
      MoveToPath := '';
      if not CopyMove then
        MessageBox (0, GetLngText (15), 'RamCopy', MB_OK or MB_ICONERROR or
          MB_TASKMODAL);
    end;
  if InfoStartEnd = FS_STATUS_END then
    if (InfoOperation >= FS_STATUS_OP_GET_SINGLE) and (InfoOperation <=
      FS_STATUS_OP_RENMOV_MULTI) then
    begin
      if CopyMove then
      begin
        if FlushRAM (False) <> FS_FILE_OK then
          ClearRAMFilesItems;
        if MoveOperation then
          RemoveCachedDirs;
        if DirCache <> nil then
        begin
          DirCache.Free;
          DirCache := nil;
        end;
        if FirstDirCache <> nil then
        begin
          FirstDirCache.Free;
          FirstDirCache := nil;
        end;
        CopyMove := False;
      end;
      CopyMoveDir := cmdNothing;
      DoneRAMCopy;
    end;
end;

exports
  FsGetDefRootName,
  FsFindFirst,
  FsFindNext,
  FsFindClose,
  FsInit,
  FsGetFile,
  FsPutFile,
  FsRenMovFile,
  FsMkDir,
  FsRemoveDir,
  FsDeleteFile,
  FsSetAttr,
  FsSetTime,
  FsExecuteFile,
  FsStatusInfo;

begin
  DisableThreadLibraryCalls (hInstance);
  DLLProc := @WFXMain;
  WFXMain (DLL_PROCESS_ATTACH);
end.
