unit packU;

interface

uses
	Windows,
	wcxhead;

function OpenArchive(var ArchiveData: TOpenArchiveData): THandle; stdcall; export;
function ReadHeader(hArcData: THandle; var HeaderData: THeaderData): integer; stdcall; export;
function ProcessFile(hArcData: THandle; Operation: integer; DestPath, DestName: PChar): integer; stdcall; export;
function CloseArchive (hArcData: THandle): integer; stdcall; export;
function GetPackerCaps: integer; stdcall; export;
procedure PackSetDefaultParams(var dps: TPackDefaultParamStruct); stdcall; export;
procedure ConfigurePacker(Parent: HWnd; DllInstance: HInst); stdcall; export;
procedure SetProcessDataProc(hArcData: THandle; AProcessDataProc: TProcessDataProc); stdcall; export;
procedure SetChangeVolProc(hArcData: THandle; var ChangeVolProc: TChangeVolProc); stdcall; export;

implementation

uses
	Classes, SysUtils, commonsU, AbZipper, AbUnzper, SBUtils, Bzip2, AbUtils;

var
	ProcessDataProc: TProcessDataProc;

type
	TArcData = class
	public
		UnZ: TAbUnzipper;
		TempTar: string;

		constructor Create(AUnZ: TAbUnzipper; ATempTar: string);
		destructor Destroy; override;
	end;

function OpenArchive(var ArchiveData: TOpenArchiveData): THandle; stdcall;
	function ExtractTarFromBZ(fn: string): string;
	const
		ToRead = 32768;
	var
		InS, OutS, DecS: TStream;
		Read: integer;
		Buf: array[0..ToRead - 1] of byte;
		tmpfn: string;
	begin
		InS := TFileStream.Create(fn, fmOpenRead);
		DecS := TBZDecompressionStream.Create(InS);
		tmpfn := TempFileName('bz2', 0);
		Result := ChangeFileExt(tmpfn, '.tar');
		RenameFile(tmpfn, Result);
		OutS := TFileStream.Create(Result, fmCreate);
		repeat
			Read := DecS.Read(Buf, ToRead);
			if Read > 0 then
				OutS.Write(Buf, Read);
		until Read < ToRead;
		DecS.Free;
		InS.Free;
		OutS.Free;
	end;
var
	UnZ: TAbUnzipper;
	ReadFrom, TempTar: string;
begin
	if AnsiLowerCase(ExtractFileExt(ArchiveData.ArcName)) = '.bz2' then
		if Pos('.tar.bz2', AnsiLowerCase(ArchiveData.ArcName)) > 0 then
			begin
				try
					ReadFrom := ExtractTarFromBZ(ArchiveData.ArcName);
					TempTar := ReadFrom;
				except
					Result := 0;
					ArchiveData.OpenResult := E_UNKNOWN_FORMAT;
					exit;
				end;
			end
		else
			begin
				Result := 0;
				ArchiveData.OpenResult := E_UNKNOWN_FORMAT;
				exit;
			end
	else
		begin
			ReadFrom := ArchiveData.ArcName;
			TempTar := '';
		end;

	UnZ := TAbUnzipper.Create(nil);
	UnZ.TarAutoHandle := True;
	try
		UnZ.OpenArchive(ReadFrom);
		UnZ.Tag := -1;
		Result := integer(TArcData.Create(UnZ, TempTar));
	except
		Result := 0;
		UnZ.Free;
		if TempTar <> '' then DeleteFile(TempTar);
		ArchiveData.OpenResult := E_UNKNOWN_FORMAT;
	end;
end;

function ReadHeader(hArcData: THandle; var HeaderData: THeaderData): integer; stdcall;
var
	UnZ: TAbUnzipper;
begin
	UnZ := TArcData(hArcData).UnZ;
	if UnZ.Tag >= UnZ.Count - 1 then
		Result := E_END_ARCHIVE
	else
		begin
			UnZ.Tag := UnZ.Tag + 1;
			StrPCopy(HeaderData.FileName, StringReplace(UnZ.Items[UnZ.Tag].FileName, '/', '\', [rfReplaceAll]));
			HeaderData.PackSize := UnZ.Items[UnZ.Tag].CompressedSize;
			HeaderData.UnpSize := UnZ.Items[UnZ.Tag].UncompressedSize;
			HeaderData.FileCRC := UnZ.Items[UnZ.Tag].CRC32;
			HeaderData.FileTime := DateTimeToFileDate(UnZ.Items[UnZ.Tag].LastModTimeAsDateTime);
			HeaderData.FileAttr := AbUnix2DosFileAttributes(UnZ.Items[UnZ.Tag].ExternalFileAttributes);
			Result := 0;
		end;
end;

function ProcessFile(hArcData: THandle; Operation: integer; DestPath, DestName: PChar): integer; stdcall;
var
	UnZ: TAbUnzipper;
	pth, fn: string;
begin
	UnZ := TArcData(hArcData).UnZ;
	if UnZ.Tag >= UnZ.Count then
		Result := E_END_ARCHIVE
	else if Operation = PK_SKIP then
		begin
			Result := 0;
			if Assigned(ProcessDataProc) then
				ProcessDataProc(PChar(UnZ.Items[UnZ.Tag].FileName), UnZ.Items[UnZ.Tag].UncompressedSize);
		end
	else
		begin
			try
				pth := DestPath;
				fn := DestName;
				if pth = '' then
					begin
						pth := ExtractFilePath(fn);
						fn := ExtractFileName(fn);
					end;
				UnZ.BaseDirectory := pth;
				UnZ.ExtractAt(UnZ.Tag, fn);
				Result := 0;
				if Assigned(ProcessDataProc) then
					ProcessDataProc(PChar(UnZ.Items[UnZ.Tag].FileName), UnZ.Items[UnZ.Tag].UncompressedSize);
			except
				Result := E_BAD_DATA;
			end;
		end;
end;

function CloseArchive (hArcData: THandle): integer; stdcall;
begin
	TArcData(hArcData).Free;
	Result := 0;
end;

function GetPackerCaps: integer; stdcall;
begin
	Result := PK_CAPS_MULTIPLE;
end;

procedure PackSetDefaultParams(var dps: TPackDefaultParamStruct); stdcall;
begin
end;

procedure ConfigurePacker(Parent: HWnd; DllInstance: HInst); stdcall;
begin
end;

procedure SetProcessDataProc(hArcData: THandle; AProcessDataProc: TProcessDataProc); stdcall;
begin
	ProcessDataProc := AProcessDataProc;
end;

procedure SetChangeVolProc(hArcData: THandle; var ChangeVolProc: TChangeVolProc); stdcall; export;
begin
end;

{ TArcData }

constructor TArcData.Create(AUnZ: TAbUnzipper; ATempTar: string);
begin
	UnZ := AUnZ;
	TempTar := ATempTar;
end;

destructor TArcData.Destroy;
begin
	UnZ.Free;
	if TempTar <> '' then DeleteFile(TempTar);
	inherited;
end;

end.

